/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Booleans;
import java.math.BigInteger;
import java.math.RoundingMode;

public final class DoubleMath {
    private static final double MIN_INT_AS_DOUBLE = -2.147483648E9;
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final double LN_2 = Math.log(2.0);
    @VisibleForTesting
    static final int MAX_FACTORIAL = 170;
    @VisibleForTesting
    static final double[] everySixteenthFactorial = new double[]{1.0, 2.0922789888E13, 2.631308369336935E35, 1.2413915592536073E61, 1.2688693218588417E89, 7.156945704626381E118, 9.916779348709496E149, 1.974506857221074E182, 3.856204823625804E215, 5.5502938327393044E249, 4.7147236359920616E284};

    static double roundIntermediate(double x10, RoundingMode mode) {
        if (!DoubleUtils.isFinite(x10)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isMathematicalInteger(x10));
                return x10;
            }
            case FLOOR: {
                if (x10 >= 0.0 || DoubleMath.isMathematicalInteger(x10)) {
                    return x10;
                }
                return x10 - 1.0;
            }
            case CEILING: {
                if (x10 <= 0.0 || DoubleMath.isMathematicalInteger(x10)) {
                    return x10;
                }
                return x10 + 1.0;
            }
            case DOWN: {
                return x10;
            }
            case UP: {
                if (DoubleMath.isMathematicalInteger(x10)) {
                    return x10;
                }
                return x10 + Math.copySign(1.0, x10);
            }
            case HALF_EVEN: {
                return Math.rint(x10);
            }
            case HALF_UP: {
                double z10 = Math.rint(x10);
                if (Math.abs(x10 - z10) == 0.5) {
                    return x10 + Math.copySign(0.5, x10);
                }
                return z10;
            }
            case HALF_DOWN: {
                double z11 = Math.rint(x10);
                if (Math.abs(x10 - z11) == 0.5) {
                    return x10;
                }
                return z11;
            }
        }
        throw new AssertionError();
    }

    public static int roundToInt(double x10, RoundingMode mode) {
        double z10 = DoubleMath.roundIntermediate(x10, mode);
        MathPreconditions.checkInRange(z10 > -2.147483649E9 & z10 < 2.147483648E9);
        return (int)z10;
    }

    public static long roundToLong(double x10, RoundingMode mode) {
        double z10 = DoubleMath.roundIntermediate(x10, mode);
        MathPreconditions.checkInRange(-9.223372036854776E18 - z10 < 1.0 & z10 < 9.223372036854776E18);
        return (long)z10;
    }

    public static BigInteger roundToBigInteger(double x10, RoundingMode mode) {
        if (-9.223372036854776E18 - (x10 = DoubleMath.roundIntermediate(x10, mode)) < 1.0 & x10 < 9.223372036854776E18) {
            return BigInteger.valueOf((long)x10);
        }
        int exponent = Math.getExponent(x10);
        long significand = DoubleUtils.getSignificand(x10);
        BigInteger result = BigInteger.valueOf(significand).shiftLeft(exponent - 52);
        return x10 < 0.0 ? result.negate() : result;
    }

    public static boolean isPowerOfTwo(double x10) {
        return x10 > 0.0 && DoubleUtils.isFinite(x10) && LongMath.isPowerOfTwo(DoubleUtils.getSignificand(x10));
    }

    public static double log2(double x10) {
        return Math.log(x10) / LN_2;
    }

    public static int log2(double x10, RoundingMode mode) {
        boolean increment;
        Preconditions.checkArgument(x10 > 0.0 && DoubleUtils.isFinite(x10), "x must be positive and finite");
        int exponent = Math.getExponent(x10);
        if (!DoubleUtils.isNormal(x10)) {
            return DoubleMath.log2(x10 * 4.503599627370496E15, mode) - 52;
        }
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isPowerOfTwo(x10));
            }
            case FLOOR: {
                increment = false;
                break;
            }
            case CEILING: {
                increment = !DoubleMath.isPowerOfTwo(x10);
                break;
            }
            case DOWN: {
                increment = exponent < 0 & !DoubleMath.isPowerOfTwo(x10);
                break;
            }
            case UP: {
                increment = exponent >= 0 & !DoubleMath.isPowerOfTwo(x10);
                break;
            }
            case HALF_EVEN: 
            case HALF_UP: 
            case HALF_DOWN: {
                double xScaled = DoubleUtils.scaleNormalize(x10);
                increment = xScaled * xScaled > 2.0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? exponent + 1 : exponent;
    }

    public static boolean isMathematicalInteger(double x10) {
        return DoubleUtils.isFinite(x10) && (x10 == 0.0 || 52 - Long.numberOfTrailingZeros(DoubleUtils.getSignificand(x10)) <= Math.getExponent(x10));
    }

    public static double factorial(int n10) {
        MathPreconditions.checkNonNegative("n", n10);
        if (n10 > 170) {
            return Double.POSITIVE_INFINITY;
        }
        double accum = 1.0;
        for (int i10 = 1 + (n10 & 0xFFFFFFF0); i10 <= n10; ++i10) {
            accum *= (double)i10;
        }
        return accum * everySixteenthFactorial[n10 >> 4];
    }

    public static boolean fuzzyEquals(double a10, double b10, double tolerance) {
        MathPreconditions.checkNonNegative("tolerance", tolerance);
        return Math.copySign(a10 - b10, 1.0) <= tolerance || a10 == b10 || a10 != a10 && b10 != b10;
    }

    public static int fuzzyCompare(double a10, double b10, double tolerance) {
        if (DoubleMath.fuzzyEquals(a10, b10, tolerance)) {
            return 0;
        }
        if (a10 < b10) {
            return -1;
        }
        if (a10 > b10) {
            return 1;
        }
        return Booleans.compare(Double.isNaN(a10), Double.isNaN(b10));
    }

    private DoubleMath() {
    }
}

