/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.net;

import com.google.common.annotations.Beta;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import javax.annotation.Nullable;

@Beta
public final class InetAddresses {
    private static final int IPV4_PART_COUNT = 4;
    private static final int IPV6_PART_COUNT = 8;
    private static final Inet4Address LOOPBACK4 = (Inet4Address)InetAddresses.forString("127.0.0.1");
    private static final Inet4Address ANY4 = (Inet4Address)InetAddresses.forString("0.0.0.0");

    private InetAddresses() {
    }

    private static Inet4Address getInet4Address(byte[] bytes) {
        Preconditions.checkArgument(bytes.length == 4, "Byte array has invalid length for an IPv4 address: %s != 4.", bytes.length);
        return (Inet4Address)InetAddresses.bytesToInetAddress(bytes);
    }

    public static InetAddress forString(String ipString) {
        byte[] addr = InetAddresses.ipStringToBytes(ipString);
        if (addr == null) {
            throw new IllegalArgumentException(String.format("'%s' is not an IP string literal.", ipString));
        }
        return InetAddresses.bytesToInetAddress(addr);
    }

    public static boolean isInetAddress(String ipString) {
        return InetAddresses.ipStringToBytes(ipString) != null;
    }

    private static byte[] ipStringToBytes(String ipString) {
        boolean hasColon = false;
        boolean hasDot = false;
        for (int i10 = 0; i10 < ipString.length(); ++i10) {
            char c10 = ipString.charAt(i10);
            if (c10 == '.') {
                hasDot = true;
                continue;
            }
            if (c10 == ':') {
                if (hasDot) {
                    return null;
                }
                hasColon = true;
                continue;
            }
            if (Character.digit(c10, 16) != -1) continue;
            return null;
        }
        if (hasColon) {
            if (hasDot && (ipString = InetAddresses.convertDottedQuadToHex(ipString)) == null) {
                return null;
            }
            return InetAddresses.textToNumericFormatV6(ipString);
        }
        if (hasDot) {
            return InetAddresses.textToNumericFormatV4(ipString);
        }
        return null;
    }

    private static byte[] textToNumericFormatV4(String ipString) {
        String[] address = ipString.split("\\.", 5);
        if (address.length != 4) {
            return null;
        }
        byte[] bytes = new byte[4];
        try {
            for (int i10 = 0; i10 < bytes.length; ++i10) {
                bytes[i10] = InetAddresses.parseOctet(address[i10]);
            }
        }
        catch (NumberFormatException ex2) {
            return null;
        }
        return bytes;
    }

    private static byte[] textToNumericFormatV6(String ipString) {
        int partsLo;
        int partsHi;
        String[] parts = ipString.split(":", 10);
        if (parts.length < 3 || parts.length > 9) {
            return null;
        }
        int skipIndex = -1;
        for (int i10 = 1; i10 < parts.length - 1; ++i10) {
            if (parts[i10].length() != 0) continue;
            if (skipIndex >= 0) {
                return null;
            }
            skipIndex = i10;
        }
        if (skipIndex >= 0) {
            partsHi = skipIndex;
            partsLo = parts.length - skipIndex - 1;
            if (parts[0].length() == 0 && --partsHi != 0) {
                return null;
            }
            if (parts[parts.length - 1].length() == 0 && --partsLo != 0) {
                return null;
            }
        } else {
            partsHi = parts.length;
            partsLo = 0;
        }
        int partsSkipped = 8 - (partsHi + partsLo);
        if (!(skipIndex < 0 ? partsSkipped == 0 : partsSkipped >= 1)) {
            return null;
        }
        ByteBuffer rawBytes = ByteBuffer.allocate(16);
        try {
            int i11;
            for (i11 = 0; i11 < partsHi; ++i11) {
                rawBytes.putShort(InetAddresses.parseHextet(parts[i11]));
            }
            for (i11 = 0; i11 < partsSkipped; ++i11) {
                rawBytes.putShort((short)0);
            }
            for (i11 = partsLo; i11 > 0; --i11) {
                rawBytes.putShort(InetAddresses.parseHextet(parts[parts.length - i11]));
            }
        }
        catch (NumberFormatException ex2) {
            return null;
        }
        return rawBytes.array();
    }

    private static String convertDottedQuadToHex(String ipString) {
        int lastColon = ipString.lastIndexOf(58);
        String initialPart = ipString.substring(0, lastColon + 1);
        String dottedQuad = ipString.substring(lastColon + 1);
        byte[] quad = InetAddresses.textToNumericFormatV4(dottedQuad);
        if (quad == null) {
            return null;
        }
        String penultimate = Integer.toHexString((quad[0] & 0xFF) << 8 | quad[1] & 0xFF);
        String ultimate = Integer.toHexString((quad[2] & 0xFF) << 8 | quad[3] & 0xFF);
        return initialPart + penultimate + ":" + ultimate;
    }

    private static byte parseOctet(String ipPart) {
        int octet = Integer.parseInt(ipPart);
        if (octet > 255 || ipPart.startsWith("0") && ipPart.length() > 1) {
            throw new NumberFormatException();
        }
        return (byte)octet;
    }

    private static short parseHextet(String ipPart) {
        int hextet = Integer.parseInt(ipPart, 16);
        if (hextet > 65535) {
            throw new NumberFormatException();
        }
        return (short)hextet;
    }

    private static InetAddress bytesToInetAddress(byte[] addr) {
        try {
            return InetAddress.getByAddress(addr);
        }
        catch (UnknownHostException e10) {
            throw new AssertionError((Object)e10);
        }
    }

    public static String toAddrString(InetAddress ip2) {
        Preconditions.checkNotNull(ip2);
        if (ip2 instanceof Inet4Address) {
            return ip2.getHostAddress();
        }
        Preconditions.checkArgument(ip2 instanceof Inet6Address);
        byte[] bytes = ip2.getAddress();
        int[] hextets = new int[8];
        for (int i10 = 0; i10 < hextets.length; ++i10) {
            hextets[i10] = Ints.fromBytes((byte)0, (byte)0, bytes[2 * i10], bytes[2 * i10 + 1]);
        }
        InetAddresses.compressLongestRunOfZeroes(hextets);
        return InetAddresses.hextetsToIPv6String(hextets);
    }

    private static void compressLongestRunOfZeroes(int[] hextets) {
        int bestRunStart = -1;
        int bestRunLength = -1;
        int runStart = -1;
        for (int i10 = 0; i10 < hextets.length + 1; ++i10) {
            if (i10 < hextets.length && hextets[i10] == 0) {
                if (runStart >= 0) continue;
                runStart = i10;
                continue;
            }
            if (runStart < 0) continue;
            int runLength = i10 - runStart;
            if (runLength > bestRunLength) {
                bestRunStart = runStart;
                bestRunLength = runLength;
            }
            runStart = -1;
        }
        if (bestRunLength >= 2) {
            Arrays.fill(hextets, bestRunStart, bestRunStart + bestRunLength, -1);
        }
    }

    private static String hextetsToIPv6String(int[] hextets) {
        StringBuilder buf = new StringBuilder(39);
        boolean lastWasNumber = false;
        for (int i10 = 0; i10 < hextets.length; ++i10) {
            boolean thisIsNumber;
            boolean bl2 = thisIsNumber = hextets[i10] >= 0;
            if (thisIsNumber) {
                if (lastWasNumber) {
                    buf.append(':');
                }
                buf.append(Integer.toHexString(hextets[i10]));
            } else if (i10 == 0 || lastWasNumber) {
                buf.append("::");
            }
            lastWasNumber = thisIsNumber;
        }
        return buf.toString();
    }

    public static String toUriString(InetAddress ip2) {
        if (ip2 instanceof Inet6Address) {
            return "[" + InetAddresses.toAddrString(ip2) + "]";
        }
        return InetAddresses.toAddrString(ip2);
    }

    public static InetAddress forUriString(String hostAddr) {
        int expectBytes;
        String ipString;
        Preconditions.checkNotNull(hostAddr);
        if (hostAddr.startsWith("[") && hostAddr.endsWith("]")) {
            ipString = hostAddr.substring(1, hostAddr.length() - 1);
            expectBytes = 16;
        } else {
            ipString = hostAddr;
            expectBytes = 4;
        }
        byte[] addr = InetAddresses.ipStringToBytes(ipString);
        if (addr == null || addr.length != expectBytes) {
            throw new IllegalArgumentException(String.format("Not a valid URI IP literal: '%s'", hostAddr));
        }
        return InetAddresses.bytesToInetAddress(addr);
    }

    public static boolean isUriInetAddress(String ipString) {
        try {
            InetAddresses.forUriString(ipString);
            return true;
        }
        catch (IllegalArgumentException e10) {
            return false;
        }
    }

    public static boolean isCompatIPv4Address(Inet6Address ip2) {
        if (!ip2.isIPv4CompatibleAddress()) {
            return false;
        }
        byte[] bytes = ip2.getAddress();
        return bytes[12] != 0 || bytes[13] != 0 || bytes[14] != 0 || bytes[15] != 0 && bytes[15] != 1;
    }

    public static Inet4Address getCompatIPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isCompatIPv4Address(ip2), "Address '%s' is not IPv4-compatible.", InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 12, 16));
    }

    public static boolean is6to4Address(Inet6Address ip2) {
        byte[] bytes = ip2.getAddress();
        return bytes[0] == 32 && bytes[1] == 2;
    }

    public static Inet4Address get6to4IPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.is6to4Address(ip2), "Address '%s' is not a 6to4 address.", InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 2, 6));
    }

    public static boolean isTeredoAddress(Inet6Address ip2) {
        byte[] bytes = ip2.getAddress();
        return bytes[0] == 32 && bytes[1] == 1 && bytes[2] == 0 && bytes[3] == 0;
    }

    public static TeredoInfo getTeredoInfo(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isTeredoAddress(ip2), "Address '%s' is not a Teredo address.", InetAddresses.toAddrString(ip2));
        byte[] bytes = ip2.getAddress();
        Inet4Address server = InetAddresses.getInet4Address(Arrays.copyOfRange(bytes, 4, 8));
        int flags = ByteStreams.newDataInput(bytes, 8).readShort() & 0xFFFF;
        int port = ~ByteStreams.newDataInput(bytes, 10).readShort() & 0xFFFF;
        byte[] clientBytes = Arrays.copyOfRange(bytes, 12, 16);
        for (int i10 = 0; i10 < clientBytes.length; ++i10) {
            clientBytes[i10] = ~clientBytes[i10];
        }
        Inet4Address client = InetAddresses.getInet4Address(clientBytes);
        return new TeredoInfo(server, client, port, flags);
    }

    public static boolean isIsatapAddress(Inet6Address ip2) {
        if (InetAddresses.isTeredoAddress(ip2)) {
            return false;
        }
        byte[] bytes = ip2.getAddress();
        if ((bytes[8] | 3) != 3) {
            return false;
        }
        return bytes[9] == 0 && bytes[10] == 94 && bytes[11] == -2;
    }

    public static Inet4Address getIsatapIPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isIsatapAddress(ip2), "Address '%s' is not an ISATAP address.", InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 12, 16));
    }

    public static boolean hasEmbeddedIPv4ClientAddress(Inet6Address ip2) {
        return InetAddresses.isCompatIPv4Address(ip2) || InetAddresses.is6to4Address(ip2) || InetAddresses.isTeredoAddress(ip2);
    }

    public static Inet4Address getEmbeddedIPv4ClientAddress(Inet6Address ip2) {
        if (InetAddresses.isCompatIPv4Address(ip2)) {
            return InetAddresses.getCompatIPv4Address(ip2);
        }
        if (InetAddresses.is6to4Address(ip2)) {
            return InetAddresses.get6to4IPv4Address(ip2);
        }
        if (InetAddresses.isTeredoAddress(ip2)) {
            return InetAddresses.getTeredoInfo(ip2).getClient();
        }
        throw new IllegalArgumentException(String.format("'%s' has no embedded IPv4 address.", InetAddresses.toAddrString(ip2)));
    }

    public static boolean isMappedIPv4Address(String ipString) {
        byte[] bytes = InetAddresses.ipStringToBytes(ipString);
        if (bytes != null && bytes.length == 16) {
            int i10;
            for (i10 = 0; i10 < 10; ++i10) {
                if (bytes[i10] == 0) continue;
                return false;
            }
            for (i10 = 10; i10 < 12; ++i10) {
                if (bytes[i10] == -1) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    public static Inet4Address getCoercedIPv4Address(InetAddress ip2) {
        if (ip2 instanceof Inet4Address) {
            return (Inet4Address)ip2;
        }
        byte[] bytes = ip2.getAddress();
        boolean leadingBytesOfZero = true;
        for (int i10 = 0; i10 < 15; ++i10) {
            if (bytes[i10] == 0) continue;
            leadingBytesOfZero = false;
            break;
        }
        if (leadingBytesOfZero && bytes[15] == 1) {
            return LOOPBACK4;
        }
        if (leadingBytesOfZero && bytes[15] == 0) {
            return ANY4;
        }
        Inet6Address ip6 = (Inet6Address)ip2;
        long addressAsLong = 0L;
        addressAsLong = InetAddresses.hasEmbeddedIPv4ClientAddress(ip6) ? (long)InetAddresses.getEmbeddedIPv4ClientAddress(ip6).hashCode() : ByteBuffer.wrap(ip6.getAddress(), 0, 8).getLong();
        int coercedHash = Hashing.murmur3_32().hashLong(addressAsLong).asInt();
        if ((coercedHash |= 0xE0000000) == -1) {
            coercedHash = -2;
        }
        return InetAddresses.getInet4Address(Ints.toByteArray(coercedHash));
    }

    public static int coerceToInteger(InetAddress ip2) {
        return ByteStreams.newDataInput(InetAddresses.getCoercedIPv4Address(ip2).getAddress()).readInt();
    }

    public static Inet4Address fromInteger(int address) {
        return InetAddresses.getInet4Address(Ints.toByteArray(address));
    }

    public static InetAddress fromLittleEndianByteArray(byte[] addr) throws UnknownHostException {
        byte[] reversed = new byte[addr.length];
        for (int i10 = 0; i10 < addr.length; ++i10) {
            reversed[i10] = addr[addr.length - i10 - 1];
        }
        return InetAddress.getByAddress(reversed);
    }

    public static InetAddress increment(InetAddress address) {
        int i10;
        byte[] addr = address.getAddress();
        for (i10 = addr.length - 1; i10 >= 0 && addr[i10] == -1; --i10) {
            addr[i10] = 0;
        }
        Preconditions.checkArgument(i10 >= 0, "Incrementing %s would wrap.", address);
        int n10 = i10;
        addr[n10] = (byte)(addr[n10] + 1);
        return InetAddresses.bytesToInetAddress(addr);
    }

    public static boolean isMaximum(InetAddress address) {
        byte[] addr = address.getAddress();
        for (int i10 = 0; i10 < addr.length; ++i10) {
            if (addr[i10] == -1) continue;
            return false;
        }
        return true;
    }

    @Beta
    public static final class TeredoInfo {
        private final Inet4Address server;
        private final Inet4Address client;
        private final int port;
        private final int flags;

        public TeredoInfo(@Nullable Inet4Address server, @Nullable Inet4Address client, int port, int flags) {
            Preconditions.checkArgument(port >= 0 && port <= 65535, "port '%s' is out of range (0 <= port <= 0xffff)", port);
            Preconditions.checkArgument(flags >= 0 && flags <= 65535, "flags '%s' is out of range (0 <= flags <= 0xffff)", flags);
            this.server = Objects.firstNonNull(server, ANY4);
            this.client = Objects.firstNonNull(client, ANY4);
            this.port = port;
            this.flags = flags;
        }

        public Inet4Address getServer() {
            return this.server;
        }

        public Inet4Address getClient() {
            return this.client;
        }

        public int getPort() {
            return this.port;
        }

        public int getFlags() {
            return this.flags;
        }
    }
}

