﻿/**********
This library is free software; you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the
Free Software Foundation; either version 2.1 of the License, or (at your
option) any later version. (See <http://www.gnu.org/copyleft/lesser.html>.)

This library is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
more details.

You should have received a copy of the GNU Lesser General Public License
along with this library; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
**********/
// "liveMedia"
// Copyright (c) 1996-2015 Live Networks, Inc.  All rights reserved.
// JPEG Video (RFC 2435) RTP Sources
// Implementation

#include "JPEGVideoRTPSource.hh"

////////// JPEGBufferedPacket and JPEGBufferedPacketFactory //////////

class JPEGBufferedPacket: public BufferedPacket {
public:
  Boolean completesFrame = false;
  bool completeExtHeader = false;
  bool noBody = false;

private:
  // Redefined virtual functions:
  virtual void reset();
  virtual unsigned nextEnclosedFrameSize(unsigned char*& framePtr,
					 unsigned dataSize);
};

class JPEGBufferedPacketFactory: public BufferedPacketFactory {
private: // redefined virtual functions
  virtual BufferedPacket* createNewPacket(MultiFramedRTPSource* ourSource);
};

void markPacketAsHeaderComplete(BufferedPacket* packet)
{
  ((JPEGBufferedPacket*)packet)->completeExtHeader = true;
}

void markPacketAsNoBody(BufferedPacket* packet)
{
  ((JPEGBufferedPacket*)packet)->noBody = true;
}

////////// JPEGVideoRTPSource implementation //////////

#define BYTE unsigned char
#define WORD unsigned
#define DWORD unsigned long

JPEGVideoRTPSource*
JPEGVideoRTPSource::createNew(UsageEnvironment& env, Groupsock* RTPgs,
			      unsigned char rtpPayloadFormat,
			      unsigned rtpTimestampFrequency,
			      unsigned defaultWidth, unsigned defaultHeight) {
  return new JPEGVideoRTPSource(env, RTPgs, rtpPayloadFormat,
				rtpTimestampFrequency, defaultWidth, defaultHeight);
}

JPEGVideoRTPSource::JPEGVideoRTPSource(UsageEnvironment& env,
				       Groupsock* RTPgs,
				       unsigned char rtpPayloadFormat,
				       unsigned rtpTimestampFrequency,
				       unsigned defaultWidth, unsigned defaultHeight)
  : MultiFramedRTPSource(env, RTPgs,
			 rtpPayloadFormat, rtpTimestampFrequency,
			 new JPEGBufferedPacketFactory),
    fDefaultWidth(defaultWidth), fDefaultHeight(defaultHeight),
    fUnits(0), fScaleX(0), fScaleY(0)
{
}

JPEGVideoRTPSource::~JPEGVideoRTPSource() {
}

enum {
	MARKER_SOF0	= 0xc0,		// start-of-frame, baseline scan
	MARKER_SOI	= 0xd8,		// start of image
	MARKER_EOI	= 0xd9,		// end of image
	MARKER_SOS	= 0xda,		// start of scan
	MARKER_DRI	= 0xdd,		// restart interval
	MARKER_DQT	= 0xdb,		// define quantization tables
	MARKER_DHT  = 0xc4,		// huffman tables
	MARKER_APP_FIRST	= 0xe0,
	MARKER_APP_LAST		= 0xef,
	MARKER_COMMENT		= 0xfe,
};

static unsigned char const lum_dc_codelens[] = {
  0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0,
};

static unsigned char const lum_dc_symbols[] = {
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11,
};

static unsigned char const lum_ac_codelens[] = {
  0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 0x7d,
};

static unsigned char const lum_ac_symbols[] = {
  0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
  0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
  0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
  0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
  0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16,
  0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
  0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
  0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
  0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
  0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
  0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
  0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
  0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
  0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
  0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
  0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
  0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4,
  0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
  0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
  0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
  0xf9, 0xfa,
};

static unsigned char const chm_dc_codelens[] = {
  0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0,
};

static unsigned char const chm_dc_symbols[] = {
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11,
};

static unsigned char const chm_ac_codelens[] = {
  0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 0x77,
};

static unsigned char const chm_ac_symbols[] = {
  0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
  0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
  0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
  0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0,
  0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34,
  0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
  0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38,
  0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
  0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
  0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
  0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
  0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96,
  0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5,
  0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
  0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
  0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2,
  0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
  0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
  0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
  0xf9, 0xfa,
};

static void createHuffmanHeader(unsigned char*& p,
				unsigned char const* codelens,
				int ncodes,
				unsigned char const* symbols,
				int nsymbols,
				int tableNo, int tableClass) {
  *p++ = 0xff; *p++ = MARKER_DHT;
  *p++ = 0;               /* length msb */
  *p++ = 3 + ncodes + nsymbols; /* length lsb */
  *p++ = (tableClass << 4) | tableNo;
  memcpy(p, codelens, ncodes);
  p += ncodes;
  memcpy(p, symbols, nsymbols);
  p += nsymbols;
}

static unsigned computeJPEGHeaderSize(unsigned qtlen, unsigned dri) {
  unsigned qtlen_half = qtlen/2; // in case qtlen is odd; shouldn't happen
  qtlen = qtlen_half*2;

  unsigned numQtables = qtlen > 64 ? 2 : 1;
  return 485 + numQtables*5 + qtlen + (dri > 0 ? 6 : 0);
}

static void createJPEGHeader(unsigned char* buf, unsigned type,
			     unsigned w, unsigned h, unsigned units, unsigned scalex, unsigned scaley,
			     unsigned char const* qtables, unsigned qtlen,
			     unsigned dri) {
  unsigned char *ptr = buf;
  unsigned numQtables = qtlen > 64 ? 2 : 1;

  // MARKER_SOI:
  *ptr++ = 0xFF; *ptr++ = MARKER_SOI;

  // MARKER_APP_FIRST:
  *ptr++ = 0xFF; *ptr++ = MARKER_APP_FIRST;
  *ptr++ = 0x00; *ptr++ = 0x10; // size of chunk
  *ptr++ = 'J'; *ptr++ = 'F'; *ptr++ = 'I'; *ptr++ = 'F'; *ptr++ = 0x00;
  *ptr++ = 0x01; *ptr++ = 0x01; // JFIF format version (1.1)
  // AS update for JPEG aspect ratio header
  if (scalex && scaley) {
    *ptr++ = units; 
    *ptr++ = 0x00; *ptr++ = scaley; // Horizontal pixel aspect ratio
    *ptr++ = 0x00; *ptr++ = scalex; // Vertical pixel aspect ratio
  } else {
    *ptr++ = 0x00; // no units
    *ptr++ = 0x00; *ptr++ = 0x01; // Horizontal pixel aspect ratio
    *ptr++ = 0x00; *ptr++ = 0x01; // Vertical pixel aspect ratio
  }
  // AS update for JPEG aspect ratio header end
  *ptr++ = 0x00; *ptr++ = 0x00; // no thumbnail
  // MARKER_DRI:
  if (dri > 0) {
    *ptr++ = 0xFF; *ptr++ = MARKER_DRI;
    *ptr++ = 0x00; *ptr++ = 0x04; // size of chunk
    *ptr++ = (BYTE)(dri >> 8); *ptr++ = (BYTE)(dri); // restart interval
  }

  // MARKER_DQT (luma):
  unsigned tableSize = numQtables == 1 ? qtlen : qtlen/2;
  *ptr++ = 0xFF; *ptr++ = MARKER_DQT;
  *ptr++ = 0x00; *ptr++ = tableSize + 3; // size of chunk
  *ptr++ = 0x00; // precision(0), table id(0)
  memcpy(ptr, qtables, tableSize);
  qtables += tableSize;
  ptr += tableSize;

  if (numQtables > 1) {
    unsigned tableSize = qtlen - qtlen/2;
    // MARKER_DQT (chroma):
    *ptr++ = 0xFF; *ptr++ = MARKER_DQT;
    *ptr++ = 0x00; *ptr++ = tableSize + 3; // size of chunk
    *ptr++ = 0x01; // precision(0), table id(1)
    memcpy(ptr, qtables, tableSize);
    qtables += tableSize;
    ptr += tableSize;
  }

  // MARKER_SOF0:
  *ptr++ = 0xFF; *ptr++ = MARKER_SOF0;
  *ptr++ = 0x00; *ptr++ = 0x11; // size of chunk
  *ptr++ = 0x08; // sample precision
  *ptr++ = (BYTE)(h >> 8);
  *ptr++ = (BYTE)(h); // number of lines (must be a multiple of 8)
  *ptr++ = (BYTE)(w >> 8);
  *ptr++ = (BYTE)(w); // number of columns (must be a multiple of 8)
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = type ? 0x22 : 0x21; // sampling ratio (h,v)
  *ptr++ = 0x00; // quant table id
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = numQtables == 1 ? 0x00 : 0x01; // quant table id
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = numQtables == 1 ? 0x00 : 0x01; // quant table id

  createHuffmanHeader(ptr, lum_dc_codelens, sizeof lum_dc_codelens,
		      lum_dc_symbols, sizeof lum_dc_symbols, 0, 0);
  createHuffmanHeader(ptr, lum_ac_codelens, sizeof lum_ac_codelens,
		      lum_ac_symbols, sizeof lum_ac_symbols, 0, 1);
  createHuffmanHeader(ptr, chm_dc_codelens, sizeof chm_dc_codelens,
		      chm_dc_symbols, sizeof chm_dc_symbols, 1, 0);
  createHuffmanHeader(ptr, chm_ac_codelens, sizeof chm_ac_codelens,
		      chm_ac_symbols, sizeof chm_ac_symbols, 1, 1);

  // MARKER_SOS:
  *ptr++ = 0xFF;  *ptr++ = MARKER_SOS;
  *ptr++ = 0x00; *ptr++ = 0x0C; // size of chunk
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = 0x00; // huffman table id (DC, AC)
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x00; // start of spectral
  *ptr++ = 0x3F; // end of spectral
  *ptr++ = 0x00; // successive approximation bit position (high, low)
}

// The default 'luma' and 'chroma' quantizer tables, in zigzag order:
static unsigned char const defaultQuantizers[128] = {
  // luma table:
  16, 11, 12, 14, 12, 10, 16, 14,
  13, 14, 18, 17, 16, 19, 24, 40,
  26, 24, 22, 22, 24, 49, 35, 37,
  29, 40, 58, 51, 61, 60, 57, 51,
  56, 55, 64, 72, 92, 78, 64, 68,
  87, 69, 55, 56, 80, 109, 81, 87,
  95, 98, 103, 104, 103, 62, 77, 113,
  121, 112, 100, 120, 92, 101, 103, 99,
  // chroma table:
  17, 18, 18, 24, 21, 24, 47, 26,
  26, 47, 99, 66, 56, 66, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99,
  99, 99, 99, 99, 99, 99, 99, 99
};

static void makeDefaultQtables(unsigned char* resultTables, unsigned Q) {
  int factor = Q;
  int q;

  if (Q < 1) factor = 1;
  else if (Q > 99) factor = 99;

  if (Q < 50) {
    q = 5000 / factor;
  } else {
    q = 200 - factor*2;
  }

  for (int i = 0; i < 128; ++i) {
    int newVal = (defaultQuantizers[i]*q + 50)/100;
    if (newVal < 1) newVal = 1;
    else if (newVal > 255) newVal = 255;
    resultTables[i] = newVal;
  }
}

void jpegExtData::Clean()
{
  memset(this, 0, sizeof(jpegExtData));
  lastBuffer = headers[0];
}

#define CALC_HEADER_SIZE(buffer) (2 + ((buffer[2] << 8) | buffer[3]))

unsigned jpegExtData::
computeJPEGHeaderSize(unsigned qtlen, unsigned dri) {


  // check integrity
  // SOF - DQT
  if (headers[2][0]) { 
    if (!headers[2][12] && !headers[2][15]) { // only one table need
      memset(headers[9], 0, sizeof(headers[9]));
    } else {
      if (!headers[9][0]) { // no second table - SOF invalid, DQT1 invalid
        //memset(headers[2], 0, sizeof(headers[2]));
        //memset(headers[8], 0, sizeof(headers[8]));
      }
    }
  } else {
    // use default QT
    memset(headers[8], 0, sizeof(headers[8]) * 2);
  };
  
  // SOS - DHT
  if (headers[3][0]) { 
  } else {
    // use default HT
    memset(headers[4], 0, sizeof(headers[4]) * 4);
  };

  // calculation
  int len = 2; // SOI

  // 0xe0: // APP0-JFIF
  if (headers[0][0]) { 
    len += CALC_HEADER_SIZE(headers[0]);
  } else {
    len += 18;
  };

  // 0xdd: // DRI
  if (headers[1][0]) { 
    len += CALC_HEADER_SIZE(headers[1]);
  } else {
    if (dri > 0) len += 6;
  };

  // 0xdb: // DQT
  if (headers[8][0] || headers[9][0]) { 
    if (headers[8][0]) { 
      len += CALC_HEADER_SIZE(headers[8]);
    }
    if (headers[9][0]) { 
      len += CALC_HEADER_SIZE(headers[9]);
    }
  } else {
    // use default QT
    len += 5 + qtlen;
    if (qtlen > 64) len += 5;
  };


  // 0xc0: // SOF0
  if (headers[2][0]) { 
    len += CALC_HEADER_SIZE(headers[2]);
  } else {
    // use default QT
    len += 19;
  };

  // 0xc4: // DHT
  if (headers[4][0]) { 
    len += CALC_HEADER_SIZE(headers[4]);
  } else if (!headers[3][0]) {
    // use default HT 
    len += 5 + sizeof(lum_dc_codelens) + sizeof(lum_dc_symbols);
  };
  // 0xc4: // DHT
  if (headers[6][0]) { 
    len += CALC_HEADER_SIZE(headers[6]);
  } else if (!headers[3][0]) {
    // use default HT 
    len += 5 + sizeof(lum_ac_codelens) + sizeof(lum_ac_symbols);
  };
  // 0xc4: // DHT
  if (headers[5][0]) { 
    len += CALC_HEADER_SIZE(headers[5]);
  } else if (!headers[3][0]) {
    // use default HT 
    len += 5 + sizeof(chm_dc_codelens) + sizeof(chm_dc_symbols);
  };
  // 0xc4: // DHT
  if (headers[7][0]) { 
    len += CALC_HEADER_SIZE(headers[7]);
  } else if (!headers[3][0]) {
    // use default HT 
    len += 5 + sizeof(chm_ac_codelens) + sizeof(chm_ac_symbols);
  };

  // 0xda: // SOS
  if (headers[3][0]) { 
    len += CALC_HEADER_SIZE(headers[3]);
  } else {
    // use default HT
    len += 14;
  };

  return len;
}

void jpegExtData::
createJPEGHeader(unsigned char* buf, unsigned type,
                      unsigned w, unsigned h,
                      unsigned char const* qtables, unsigned qtlen,
                      unsigned dri) {

  unsigned char *ptr = buf;
  unsigned numQtables = qtlen > 64 ? 2 : 1;
  unsigned int len;

  // MARKER_SOI:
  *ptr++ = 0xFF; *ptr++ = MARKER_SOI;

  if (headers[0][0]) { 
    len = CALC_HEADER_SIZE(headers[0]);
    memcpy(ptr, headers[0], len);
    ptr += len;
  } else {
  // MARKER_APP_FIRST:
  *ptr++ = 0xFF; *ptr++ = MARKER_APP_FIRST;
  *ptr++ = 0x00; *ptr++ = 0x10; // size of chunk
  *ptr++ = 'J'; *ptr++ = 'F'; *ptr++ = 'I'; *ptr++ = 'F'; *ptr++ = 0x00;
  *ptr++ = 0x01; *ptr++ = 0x01; // JFIF format version (1.1)
  *ptr++ = 0x00; // no units
  *ptr++ = 0x00; *ptr++ = 0x01; // Horizontal pixel aspect ratio
  *ptr++ = 0x00; *ptr++ = 0x01; // Vertical pixel aspect ratio
  *ptr++ = 0x00; *ptr++ = 0x00; // no thumbnail
  };

  // 0xdd: // DRI
  if (headers[1][0]) { 
    len = CALC_HEADER_SIZE(headers[1]);
    memcpy(ptr, headers[1], len);
    ptr += len;
  } else {
  // MARKER_DRI:
  if (dri > 0) {
    *ptr++ = 0xFF; *ptr++ = MARKER_DRI;
    *ptr++ = 0x00; *ptr++ = 0x04; // size of chunk
    *ptr++ = (BYTE)(dri >> 8); *ptr++ = (BYTE)(dri); // restart interval
  }
  };

  // 0xc0: // SOF0
  if (headers[2][0]) { 
    len = CALC_HEADER_SIZE(headers[8]);
    memcpy(ptr, headers[8], len);
    ptr += len;
    if (headers[9][0]) {
      len = CALC_HEADER_SIZE(headers[9]);
      memcpy(ptr, headers[9], len);
      ptr += len;
    }
    len = CALC_HEADER_SIZE(headers[2]);
    memcpy(ptr, headers[2], len);
    ptr += len;
  } else {
  // MARKER_DQT (luma):
  unsigned tableSize = numQtables == 1 ? qtlen : qtlen/2;
  *ptr++ = 0xFF; *ptr++ = MARKER_DQT;
  *ptr++ = 0x00; *ptr++ = tableSize + 3; // size of chunk
  *ptr++ = 0x00; // precision(0), table id(0)
  memcpy(ptr, qtables, tableSize);
  qtables += tableSize;
  ptr += tableSize;

  if (numQtables > 1) {
    unsigned tableSize = qtlen - qtlen/2;
    // MARKER_DQT (chroma):
    *ptr++ = 0xFF; *ptr++ = MARKER_DQT;
    *ptr++ = 0x00; *ptr++ = tableSize + 3; // size of chunk
    *ptr++ = 0x01; // precision(0), table id(1)
    memcpy(ptr, qtables, tableSize);
    qtables += tableSize;
    ptr += tableSize;
  }

  // MARKER_SOF0:
  *ptr++ = 0xFF; *ptr++ = MARKER_SOF0;
  *ptr++ = 0x00; *ptr++ = 0x11; // size of chunk
  *ptr++ = 0x08; // sample precision
  *ptr++ = (BYTE)(h >> 8);
  *ptr++ = (BYTE)(h); // number of lines (must be a multiple of 8)
  *ptr++ = (BYTE)(w >> 8);
  *ptr++ = (BYTE)(w); // number of columns (must be a multiple of 8)
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = type ? 0x22 : 0x21; // sampling ratio (h,v)
  *ptr++ = 0x00; // quant table id
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = numQtables == 1 ? 0x00 : 0x01; // quant table id
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = 0x01; // quant table id
  }

  // 0xda: // SOS
  if (headers[3][0]) { 
    if (headers[4][0]) {
      len = CALC_HEADER_SIZE(headers[4]);
      memcpy(ptr, headers[4], len);
      ptr += len;
    }
    if (headers[6][0]) {
      len = CALC_HEADER_SIZE(headers[6]);
      memcpy(ptr, headers[6], len);
      ptr += len;
    }
    if (headers[5][0]) {
      len = CALC_HEADER_SIZE(headers[5]);
      memcpy(ptr, headers[5], len);
      ptr += len;
    }
    if (headers[7][0]) {
      len = CALC_HEADER_SIZE(headers[7]);
      memcpy(ptr, headers[7], len);
      ptr += len;
    }
    len = CALC_HEADER_SIZE(headers[3]);
    memcpy(ptr, headers[3], len);
    ptr += len;
  } else {
  createHuffmanHeader(ptr, lum_dc_codelens, sizeof lum_dc_codelens,
		      lum_dc_symbols, sizeof lum_dc_symbols, 0, 0);
  createHuffmanHeader(ptr, lum_ac_codelens, sizeof lum_ac_codelens,
		      lum_ac_symbols, sizeof lum_ac_symbols, 0, 1);
  createHuffmanHeader(ptr, chm_dc_codelens, sizeof chm_dc_codelens,
		      chm_dc_symbols, sizeof chm_dc_symbols, 1, 0);
  createHuffmanHeader(ptr, chm_ac_codelens, sizeof chm_ac_codelens,
		      chm_ac_symbols, sizeof chm_ac_symbols, 1, 1);

  // MARKER_SOS:
  *ptr++ = 0xFF;  *ptr++ = MARKER_SOS;
  *ptr++ = 0x00; *ptr++ = 0x0C; // size of chunk
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = 0x00; // huffman table id (DC, AC)
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x00; // start of spectral
  *ptr++ = 0x3F; // end of spectral
  *ptr++ = 0x00; // successive approximation bit position (high, low)
  }
}

#define FILL_HEADER(n)                                    \
  if (fExtData.LastSize > 0) {                            \
    int rem = fExtData.LastSize;                          \
    memcpy(fExtData.headers[n], buffer, rem);             \
    buffer = bPacket->data();                             \
    memcpy(fExtData.headers[n] + rem, buffer, len - rem); \
    fExtData.LastSize = 0;                                \
  } else {                                                \
    memcpy(fExtData.headers[n], buffer, len);             \
  }                                                       \
  fExtData.lastBuffer = fExtData.headers[n]



Boolean JPEGVideoRTPSource
::processExtData(BufferedPacket* bPacket, unsigned extHdr) 
{
#if 1
  return context.nextPacketExt(bPacket, extHdr);
#else
  // general (non 3MP) image but with JFIF
  if (((DWORD*)bPacket->data())[1] == 0x4649464a) {
    fUnits = bPacket->data()[11];
    fScaleX = bPacket->data()[13];
    fScaleY = bPacket->data()[15];
  }

/// AS - Possible fix for Flir resolution errata
#if 0
  if (((extHdr >> 16) != 0xFFd8) &&
      ((extHdr >> 16) != 0xFFff)) {
    return False;
  }
#endif

  int size = 4*(extHdr&0xFFFF);
  const unsigned char* buffer = bPacket->data();

  // image data in plain extension, not in payload (or at least we will skip it)
  if (fExtData.UseExtAsPayload) {
    // very dirty HACK
    bPacket->removePadding(bPacket->dataSize() - 4*(extHdr&0xFFFF));
    fExtData.Ofs = fExtData.Size;
    fExtData.Size += bPacket->dataSize();
    return True;
  }

  if (fExtData.AllProcessed) {
    return False; // this is not first packet
  }

  if (fExtData.LastSize < 0) {
    // there were missed bytes on some header from previous packet
    // add them 
    if(sizeof(buffer) > sizeof(fExtData.lastBuffer) && -fExtData.LastSize > sizeof(fExtData.lastBuffer)){
		  //ERROR
		//memcpy(fExtData.lastBuffer, buffer, -fExtData.LastSize);

	  } else if(-fExtData.LastSize > sizeof(buffer)){
		  //ERROR
		  //memcpy(fExtData.lastBuffer, buffer, -fExtData.LastSize);

	  } else {

		  memcpy(fExtData.lastBuffer, buffer, -fExtData.LastSize);
	  }
    buffer += (-fExtData.LastSize);
    size -= (-fExtData.LastSize);
    fExtData.LastSize = 0;
  } else if (fExtData.LastSize > 0) {
    buffer = fExtData.unparsed;
    size += fExtData.LastSize;
  }

  while (size > 0) {
    
    if (*buffer != 0xff) { // we have plain data!!!
      fExtData.AllProcessed = true;
      fExtData.UseExtAsPayload = true;
      bPacket->removePadding(bPacket->dataSize() - 4*(extHdr&0xFFFF));
      bPacket->skip(4*(extHdr&0xFFFF) - size);
      fExtData.Ofs = fExtData.Size;
      fExtData.Size += bPacket->dataSize();
      return true;
    }

    if (size < 4)
      break;

    // calculate ext part len
    int len = CALC_HEADER_SIZE(buffer);
    switch (buffer[1]) {
      case 0xe0: // APP0-JFIF
        FILL_HEADER(0);
        if (0) // we will use it if there is a bug on BOSCH side
        { 
          unsigned char c = fExtData.headers[0][15];
          fExtData.headers[0][15] = fExtData.headers[0][13];
          fExtData.headers[0][13] = c;
        }
        break;
      case 0xdd: // DRI
        FILL_HEADER(1);
        break;
      case 0xc0: // SOF0
        FILL_HEADER(2);
        break;
      case 0xda: // SOS
        FILL_HEADER(3);
        break;
      case 0xc4: // DHT
        {
          int pos = buffer[4];
          pos =(pos >> 3) | (pos & 15);
          if (pos < 4) {
            FILL_HEADER(4 + pos);
          }
        }
        break;
      case 0xdb: // DQT
        {
          int pos = buffer[4];
          if (pos < 2) {
            FILL_HEADER(8 + pos);
          }
        }
        break;
      default:;   // do nothing, skip
    }
    buffer += len;
    size -= len;

    if (size < 0) {
      fExtData.lastBuffer += (len + size);
    }
  }

  fExtData.LastSize = size;
  // save unparsed data
  while (size > 0) {
    fExtData.unparsed[fExtData.LastSize - size--] = *buffer++;
  }
  return False;
#endif
}
int fullsize = 0;
Boolean JPEGVideoRTPSource
::processSpecialHeader(BufferedPacket* packet,
		       unsigned& resultSpecialHeaderSize) {

  unsigned char* headerStart = packet->data();
  unsigned packetSize = packet->dataSize();
  fullsize += packetSize;
  if (((JPEGBufferedPacket*)packet)->completeExtHeader) {
    context.pushHeader(packet, resultSpecialHeaderSize);
    ((JPEGBufferedPacket*)packet)->completeExtHeader = false;
    fCurrentPacketBeginsFrame = true;
    fCurrentPacketCompletesFrame = false;

    context.getResolution(fFrameWidth, fFrameHeight);

    return true;
  }

  if (((JPEGBufferedPacket*)packet)->noBody) {
    fCurrentPacketBeginsFrame = false;
    fCurrentPacketCompletesFrame = false;
    if (packet->rtpMarkerBit()) {
      ((JPEGBufferedPacket*)packet)->completesFrame
        = fCurrentPacketCompletesFrame = packet->rtpMarkerBit();
      context.frameDone();
      fullsize = 0;
    }
    resultSpecialHeaderSize = 0;
    return true;
  }
    
  unsigned char* qtables = NULL;
  unsigned qtlen = 0;
  unsigned dri = 0;

  // There's at least 8-byte video-specific header
  /*
0                   1                   2                   3
0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
| Type-specific |              Fragment Offset                  |
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
|      Type     |       Q       |     Width     |     Height    |
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
  */
  if (packetSize < 8) return False;

  resultSpecialHeaderSize = 8;

  unsigned Offset = (unsigned)((DWORD)headerStart[1] << 16 | (DWORD)headerStart[2] << 8 | (DWORD)headerStart[3]);
  unsigned Type = (unsigned)headerStart[4];
  unsigned type = Type & 1;
  unsigned Q = (unsigned)headerStart[5];
  unsigned width = (unsigned)headerStart[6] * 8;
  unsigned height = (unsigned)headerStart[7] * 8;

#if 1
  context.nextPacketFrame(packet, resultSpecialHeaderSize);
  if (0 == Offset) {
    context.getResolution(fFrameWidth, fFrameHeight);
  }
#else

  if (fDefaultWidth != 0 && fDefaultHeight != 0 && !fExtData.headers[2][0]) {
    // Use the default width and height parameters instead:
    width = fDefaultWidth;
    height = fDefaultHeight;
  }
  // check extension's SOF-section for width and height parameters
  if (fExtData.headers[2][0]) {
    fFrameHeight = (fExtData.headers[2][5] << 8) + fExtData.headers[2][6];
    fFrameWidth = (fExtData.headers[2][7] << 8) + fExtData.headers[2][8];
    height = fFrameHeight;
    width = fFrameWidth;
  } else {
		/// AS fix for wide resolution error for 3MP+ MJPEG
		if (!fFrameWidth && !fFrameHeight) {
	    fFrameWidth = width;
      fFrameHeight = height;
		}
  }

  if (width == 0) width = 256*8; // special case
  if (height == 0) height = 256*8; // special case

  if (fExtData.UseExtAsPayload) {
    Offset = fExtData.Ofs;
    Type = type = 0;
    Q = 0;
    width = 0;
    height = 0;
    resultSpecialHeaderSize = 0;
  }

  if (Type > 63) {
    // Restart Marker header present
    /*
0                   1                   2                   3
0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
|       Restart Interval        |F|L|       Restart Count       |
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
    */
    if (packetSize < resultSpecialHeaderSize + 4) return False;

    unsigned RestartInterval = (unsigned)((WORD)headerStart[resultSpecialHeaderSize] << 8 | (WORD)headerStart[resultSpecialHeaderSize + 1]);
    dri = RestartInterval;
    resultSpecialHeaderSize += 4;
  }

  if (Offset == 0) {
    if (Q > 127) {
      // Quantization Table header present
/*
0                   1                   2                   3
0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
|      MBZ      |   Precision   |             Length            |
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
|                    Quantization Table Data                    |
|                              ...                              |
+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
*/
      if (packetSize < resultSpecialHeaderSize + 4) return False;

      unsigned MBZ = (unsigned)headerStart[resultSpecialHeaderSize];
      if (MBZ == 0) {
	// unsigned Precision = (unsigned)headerStart[resultSpecialHeaderSize + 1];
	unsigned Length = (unsigned)((WORD)headerStart[resultSpecialHeaderSize + 2] << 8 | (WORD)headerStart[resultSpecialHeaderSize + 3]);

	//ASSERT(Length == 128);

	resultSpecialHeaderSize += 4;

	if (packetSize < resultSpecialHeaderSize + Length) return False;

	qtlen = Length;
	qtables = &headerStart[resultSpecialHeaderSize];

	resultSpecialHeaderSize += Length;
      }
    }
  }

  // If this is the first (or only) fragment of a JPEG frame, then we need
  // to synthesize a JPEG header, and prepend it to the incoming data.
  // Hack: We can do this because we allowed space for it in
  // our special "JPEGBufferedPacket" subclass.  We also adjust
  // "resultSpecialHeaderSize" to compensate for this, by subtracting
  // the size of the synthesized header.  Note that this will cause
  // "resultSpecialHeaderSize" to become negative, but the code that called
  // us (in "MultiFramedRTPSource") will handle this properly.
  if (Offset == 0) {
    unsigned char newQtables[128];
    if (qtlen == 0) {
      // A quantization table was not present in the RTP JPEG header,
      // so use the default tables, scaled according to the "Q" factor:
      makeDefaultQtables(newQtables, Q);
      qtables = newQtables;
      qtlen = sizeof newQtables;
    }

    if (fExtData.UseExtAsPayload) {
      unsigned hdrlen = fExtData.computeJPEGHeaderSize(qtlen, dri);
      resultSpecialHeaderSize -= hdrlen; // goes negative
      headerStart += (int)resultSpecialHeaderSize; // goes backward
      fExtData.createJPEGHeader(headerStart, type, width, height, qtables, qtlen, dri);
    } else {
      unsigned hdrlen = computeJPEGHeaderSize(qtlen, dri);
      resultSpecialHeaderSize -= hdrlen; // goes negative
      headerStart += (int)resultSpecialHeaderSize; // goes backward
      createJPEGHeader(headerStart, type, width, height, fUnits, fScaleX, fScaleY, qtables, qtlen, dri);
    }
  }
#endif
  fCurrentPacketBeginsFrame = (Offset == 0);

  // The RTP "M" (marker) bit indicates the last fragment of a frame:
  ((JPEGBufferedPacket*)packet)->completesFrame
   = fCurrentPacketCompletesFrame = packet->rtpMarkerBit();

  if (packet->rtpMarkerBit()) {
    fExtData.Clean();
  }
  return True;
}

char const* JPEGVideoRTPSource::MIMEtype() const {
  return "video/JPEG";
}

////////// JPEGBufferedPacket and JPEGBufferedPacketFactory implementation

void JPEGBufferedPacket::reset() {
  BufferedPacket::reset();

  // Move our "fHead" and "fTail" forward, to allow space for a synthesized
  // JPEG header to precede the RTP data that comes in over the network.
  unsigned offset = MAX_JPEG_HEADER_SIZE;
  if (offset > fPacketSize) offset = fPacketSize; // shouldn't happen
  fHead = fTail = offset;
}

unsigned JPEGBufferedPacket
::nextEnclosedFrameSize(unsigned char*& framePtr, unsigned dataSize) {
  // Normally, the enclosed frame size is just "dataSize".  If, however,
  // the frame does not end with the "EOI" marker, then add this now:
  if (completesFrame && dataSize >= 2 &&
      !(framePtr[dataSize-2] == 0xFF && framePtr[dataSize-1] == MARKER_EOI)) {
    //framePtr[dataSize++] = 0xFF;
    //framePtr[dataSize++] = MARKER_EOI;
  }
  return dataSize;
}

BufferedPacket* JPEGBufferedPacketFactory
::createNewPacket(MultiFramedRTPSource* /*ourSource*/) {
  return new JPEGBufferedPacket;
}

void ConstructQuantization(JpegSegment& segment0, JpegSegment& segment1, int quality)
{
  unsigned char newQtables[128];
  makeDefaultQtables(newQtables, quality);
  CopyQuantization(segment0, segment1, newQtables);
  segment0.setStatus(JpegSegment::Recalculated);
  segment1.setStatus(JpegSegment::Recalculated);
}

static void createHuffmanHeader(JpegSegment& segment,
  unsigned char const* codelens,
  int ncodes,
  unsigned char const* symbols,
  int nsymbols,
  int tableNo, int tableClass) {
  unsigned char b;
  segment.initForAdd(MARKER_DHT, 3 + ncodes + nsymbols);
  b = (tableClass << 4) | tableNo;
  segment.addData(&b, 1);
  segment.addData(codelens, ncodes);
  segment.addData(symbols, nsymbols);
  segment.setStatus(JpegSegment::Recalculated);
}

void ConstructHuffman(JpegSegment& segment0, JpegSegment& segment1, JpegSegment& segment2, JpegSegment& segment3)
{
  createHuffmanHeader(segment0, lum_dc_codelens, sizeof lum_dc_codelens,
    lum_dc_symbols, sizeof lum_dc_symbols, 0, 0);
  createHuffmanHeader(segment1, lum_ac_codelens, sizeof lum_ac_codelens,
    lum_ac_symbols, sizeof lum_ac_symbols, 0, 1);
  createHuffmanHeader(segment2, chm_dc_codelens, sizeof chm_dc_codelens,
    chm_dc_symbols, sizeof chm_dc_symbols, 1, 0);
  createHuffmanHeader(segment3, chm_ac_codelens, sizeof chm_ac_codelens,
    chm_ac_symbols, sizeof chm_ac_symbols, 1, 1);
}
