﻿


#include "JpegExtension.h"


enum {
  MARKER_SOF0 = 0xc0,		// start-of-frame, baseline scan
  MARKER_SOI = 0xd8,		// start of image
  MARKER_EOI = 0xd9,		// end of image
  MARKER_SOS = 0xda,		// start of scan
  MARKER_DRI = 0xdd,		// restart interval
  MARKER_DQT = 0xdb,		// define quantization tables
  MARKER_DHT = 0xc4,		// huffman tables
  MARKER_APP_FIRST = 0xe0,
  MARKER_APP_LAST = 0xef,
  MARKER_COMMENT = 0xfe,
};



JpegSegment::JpegSegment()
{
  reserve(256);
}

int JpegSegment::getData(unsigned char*& p)
{
  memcpy(p, &front(), size());
  p += size();
  return size();
}

int JpegSegment::initForAdd(int type, int size)
{
  if (size > 255) return 0;
  if (size < 2) return 0;
  resize(4);
  at(0) = 0xFF;
  at(1) = type;
  at(2) = 0;
  at(3) = size;
  waiting_ = size - 2;
  return 4;
}

unsigned char* JpegSegment::reserveForAccess(int type, int size)
{
  if (size > 255) return nullptr;
  if (size < 2) return nullptr;
  resize(size + 2);
  at(0) = 0xFF;
  at(1) = type;
  at(2) = 0;
  at(3) = size;
  waiting_ = 0;
  return &front() + 4;

}

int JpegSegment::copyForAdd(const unsigned char header[4], const unsigned char* p, int size)
{
  if (0xff != header[0]) return 0;
  int s = header[3];
  if (s > 255) return 0;
  if (s < 2) return 0;
  if (s - 2 > size) {
    waiting_ = s - 2 - size;
  }
  else {
    waiting_ = 0;
  }
  assign(header, header + 4);
  insert(end(), p, p + s - 2 - waiting_);
  return s + 2 - waiting_;
}

int JpegSegment::addData(const unsigned char* p, int size)
{
  int s = waiting_ <= size ? waiting_ : size;
  insert(end(), p, p + s);
  waiting_ -= s;
  return s;
}

int JpegSegment::getType() const
{
  return at(1);
}
int JpegSegment::getSize() const
{
  return at(3);
}

void JpegSegment::frameDone()
{
  if (Recalculated == status_ || Received == status_) {
    status_ = Backup;
  };
}

void JpegSegment::clean()
{
  clear();
  status_ = Empty;
}



/*
for each frame, each packet is scanned by live555
in case of ext bit in rtp header, nextPacketExt called - it can eat incoming data
for each packet, nextPacketFrame is called - it can extend or eat incoming data

ext data to be parsed and known segments stored

*/


// not the full packet
bool JpegContext::nextPacketExt(BufferedPacket* packet, unsigned header)
{
  //return false;
  switch (header >> 16) {
  case 0xFFD8:
  case 0xFFFF:
    return nextPacketFFF8(packet, header);
  case 0xABAC:
    return nextPacketABAC(packet, header);
  default:
    packet->skip(header & 0xFFFF);
    return true;
  }
}

void markPacketAsHeaderComplete(BufferedPacket* packet);
void markPacketAsNoBody(BufferedPacket* packet);


bool JpegContext::nextPacketFFF8(BufferedPacket* packet, unsigned header)
{
  //if ((header >> 16) == 0xFFD8 != frameDone_) {
    // 0xFFD8 should be in the firset frame
    //return false;
  //}
  int size = header & 0xFFFF;
  //if (frameDone) {
    // ok, first packet

  //}
  if (headersDone_) {
    packet->removePadding(packet->dataSize() - 4 * size);
    markPacketAsNoBody(packet);
    return true;
  };

  int s = 4 * size - passJpegSegments(packet, packet->data(), size * 4);
  if (headersDone_) {
    packet->removePadding(packet->dataSize() - 4 * size);
    markPacketAsNoBody(packet);
  }
  if (s > 0) {
    packet->skip(s);
  }
  return true;
}

int JpegContext::passJpegSegments(BufferedPacket* packet, const unsigned char* p, int size)
{
  if (headerSize) {
    int s = completeJpegHeader(p, size);
    p += s;
    size -= s;
  }

  if (waitingOtherSegment_) {
    // let complete unknown segment
    if (waitingOtherSegment_ <= size) {
      p += waitingOtherSegment_;
      size -= waitingOtherSegment_;
      waitingOtherSegment_ = 0;
    }
    else {
      waitingOtherSegment_ -= size;
      // ok, no more data, skip furhter processing for this round
      return 0;
    }
  }

  if (unfinished) {
    // let complete known segment
    int s = unfinished->addData(p, size);
    p += s;
    size -= s;
    if (!unfinished->waiting()) {
      unfinished->setStatus(JpegSegment::Received);
      unfinished = nullptr;
    }
  }

  while (size > 0) {
    if (0xFF != p[0]) {
      headersDone_ = true;
      markPacketAsHeaderComplete(packet);
      break;
    }
    int s = completeJpegHeader(p, size);
    p += s;
    size -= s;
  }
  return size;
}

int JpegContext::completeJpegHeader(const unsigned char* p, int size)
{
  if (headerSize + size > 4) {
    int s = 4 - headerSize;
    memcpy(header + headerSize, p, s);
    p += s;
    size -= s;
    // ok, segment header complete, try to parse segment
    s = passJpegSegment(header, p, size);
    headerSize = 0;
    return s;
  }
  else {
    memcpy(header + headerSize, p, size);
    headerSize += size;
    // ok, no more data, skip furhter processing for this round
    return size;
  }
}

int JpegContext::passJpegSegment(const unsigned char* h, const unsigned char* p, int size)
{
  if (0xFF != h[0]) return 0;
  unfinished = nullptr;
  switch (h[1]) {
  case MARKER_APP_FIRST: // APP0-JFIF
    unfinished = &App0;
    break;
  case MARKER_DRI: // DRI
    unfinished = &Dri;
    break;
  case MARKER_SOF0: // SOF0
    unfinished = &Sof;
    break;
  case 0xc2: // SOF2
    unfinished = &Sof;
    break;
  case MARKER_SOS: // SOS
    unfinished = &Sos;
    break;
  case MARKER_DHT: // DHT
  {
    int pos = p[0];
    pos = (pos >> 3) | (pos & 15);
    if (pos < 4) {
      unfinished = &Dht[pos];
    }
  }
  break;
  case MARKER_DQT: // DQT
  {
    int pos = p[0];
    if (pos < 2) {
      unfinished = &Dqt[pos];
    }
  }
  break;
  default:;
  }
  int s = h[3] + h[2] * 256;
  if (unfinished) {
    unfinished->setStatus(JpegSegment::Empty);
    int ret = unfinished->copyForAdd(h, p, size);
    if (!unfinished->waiting()) {
      unfinished->setStatus(JpegSegment::Received);
      unfinished = nullptr;
    }
    return ret;
  }
  else {
    if (s - 2 < size) {
      return s + 2;
    }
    else {
      waitingOtherSegment_ = s - 2 - size;
      return size + 4;
    }
  }
}

bool JpegContext::nextPacketABAC(BufferedPacket* packet, unsigned header)
{
  return false;
}


bool JpegContext::nextPacketFrame(BufferedPacket* packet, unsigned& resultSpecialHeaderSize)
{
  unsigned char* headerStart = packet->data();
  unsigned packetSize = packet->dataSize();

  unsigned Offset = (unsigned)((DWORD)headerStart[1] << 16 | (DWORD)headerStart[2] << 8 | (DWORD)headerStart[3]);
  rtp.type = (unsigned)headerStart[4];
  unsigned type = rtp.type & 1;
  rtp.quant = (unsigned)headerStart[5];
  rtp.width = (unsigned)headerStart[6] * 8;
  rtp.height = (unsigned)headerStart[7] * 8;
  rtp.dri = 0;

  if (rtp.type > 63) {
    if (packetSize < resultSpecialHeaderSize + 4) return false;
    unsigned RestartInterval = (unsigned)((WORD)headerStart[resultSpecialHeaderSize] << 8 | (WORD)headerStart[resultSpecialHeaderSize + 1]);
    rtp.dri = RestartInterval;
    resultSpecialHeaderSize += 4;
  }

  if (Offset == 0) {
    if (rtp.quant > 127) {
      // copy
      if (packetSize < resultSpecialHeaderSize + 4) return false;

      unsigned MBZ = (unsigned)headerStart[resultSpecialHeaderSize];
      if (MBZ == 0) {
        unsigned Length = (unsigned)((WORD)headerStart[resultSpecialHeaderSize + 2] << 8 | (WORD)headerStart[resultSpecialHeaderSize + 3]);
        resultSpecialHeaderSize += 4;
        if (packetSize < resultSpecialHeaderSize + Length) return false;
        if (Dqt[0].getStatus() != JpegSegment::Received) {
          CopyQuantization(Dqt[0], Dqt[1], &headerStart[resultSpecialHeaderSize]);
        }
        resultSpecialHeaderSize += Length;
      }
    }
    else {
      // create
      if (Dqt[0].getStatus() != JpegSegment::Received) {
        ConstructQuantization(Dqt[0], Dqt[1], rtp.quant);
      }
    }
    if (packetSize > resultSpecialHeaderSize) {
      formHeaders(packet, resultSpecialHeaderSize, rtp.type, rtp.quant, rtp.width, rtp.height, rtp.dri);
      headerDone();
    }
  }

  if (packet->rtpMarkerBit()) {
//    frameDone();
  }
  return false;
}

bool JpegContext::pushHeader(BufferedPacket* packet, unsigned& resultSpecialHeaderSize)
{
  resultSpecialHeaderSize = 0;
  formHeaders(packet, resultSpecialHeaderSize, rtp.type, rtp.quant, rtp.width, rtp.height, rtp.dri);
  return true;
}

void JpegContext::frameDone()
{
  unfinished = nullptr; 
  headerSize = 0;                
  waitingOtherSegment_ = 0;    
  headersDone_ = false;
  headerDone();
}

void JpegContext::getResolution(unsigned& fFrameWidth, unsigned& fFrameHeight)
{
  if (Sof.getStatus() != JpegSegment::Empty) {
    fFrameHeight = (unsigned(Sof[5]) << 8) + Sof[6];
    fFrameWidth =  (unsigned(Sof[7]) << 8) + Sof[8];
  }
}

int JpegContext::formHeaders(BufferedPacket* packet, unsigned& resultSpecialHeaderSize, int type, int Q, int width, int height, int dri)
{
  if (
    Dht[0].getStatus() == JpegSegment::Empty ||
    Dht[1].getStatus() == JpegSegment::Empty ||
    Dht[2].getStatus() == JpegSegment::Empty ||
    Dht[3].getStatus() == JpegSegment::Empty) {
    ConstructHuffman(Dht[0], Dht[1], Dht[2], Dht[3]);
  }

  if (
    Dqt[0].getStatus() == JpegSegment::Empty) {
    ConstructQuantization(Dqt[0], Dqt[1], rtp.quant);
  }

  if (App0.getStatus() == JpegSegment::Empty) {
    ConstructApp0(App0);
  }

  if (Sof.getStatus() == JpegSegment::Empty) {
    ConstructSof(Sof, width, height, type, Dqt[1].getStatus() == JpegSegment::Empty ? 1 : 2);
  }
  if (Sos.getStatus() == JpegSegment::Empty) {
    ConstructSos(Sos);
  }

  int size = 2;
  size += App0.size();
  size += Dri.size();
  size += Sof.size();
  size += Sos.size();
  for (int i = 0; i < 4; ++i) size += Dht[i].size();
  for (int i = 0; i < 2; ++i) size += Dqt[i].size();

  resultSpecialHeaderSize -= size;
  unsigned char* headerStart = packet->data() + (int)resultSpecialHeaderSize;
  *headerStart++ = 0xFF;
  *headerStart++ = MARKER_SOI;
  App0.getData(headerStart);
  if (Dri.size()) {
    Dri.getData(headerStart);
  }
  Dqt[0].getData(headerStart);
  if (Dqt[1].size()) {
    Dqt[1].getData(headerStart);
  }
  Sof.getData(headerStart);
  Dht[0].getData(headerStart);
  Dht[2].getData(headerStart);
  Dht[1].getData(headerStart);
  Dht[3].getData(headerStart);
  Sos.getData(headerStart);

  return size;
}

void JpegContext::headerDone()
{
  App0.frameDone();
  Dri.frameDone();
  Dqt[0].frameDone();
  Dqt[1].frameDone();
  Sof.frameDone();
  Dht[0].frameDone();
  Dht[1].frameDone();
  Dht[2].frameDone();
  Dht[3].frameDone();
  Sos.frameDone();
  //frameDone_ = true;
}

void ConstructApp0(JpegSegment& segment)
{
  unsigned char *ptr = segment.reserveForAccess(MARKER_APP_FIRST, 16);
  *ptr++ = 'J'; *ptr++ = 'F'; *ptr++ = 'I'; *ptr++ = 'F'; *ptr++ = 0x00;
  *ptr++ = 0x01; *ptr++ = 0x01; // JFIF format version (1.1)
  *ptr++ = 0x00; // no units
  *ptr++ = 0x00; *ptr++ = 0x01; // Horizontal pixel aspect ratio
  *ptr++ = 0x00; *ptr++ = 0x01; // Vertical pixel aspect ratio
  *ptr++ = 0x00; *ptr++ = 0x00; // no thumbnail
  segment.setStatus(JpegSegment::Recalculated);
}

void ConstructDri(JpegSegment& segment, int dri)
{
  if (dri > 0) {
    segment.reserveForAccess(MARKER_DRI, 4);
    segment[4] = (BYTE)(dri >> 8);
    segment[5] = (BYTE)(dri);
    segment.setStatus(JpegSegment::Recalculated);
  }
  else {
    segment.clean();
    segment.setStatus(JpegSegment::Empty);
  }
}

void ConstructSof(JpegSegment& segment, int w, int h, int type, int numQtables)
{
  unsigned char *ptr = segment.reserveForAccess(MARKER_SOF0, 17);
  *ptr++ = 0x08; // sample precision
  *ptr++ = (BYTE)(h >> 8);
  *ptr++ = (BYTE)(h); // number of lines (must be a multiple of 8)
  *ptr++ = (BYTE)(w >> 8);
  *ptr++ = (BYTE)(w); // number of columns (must be a multiple of 8)
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = type ? 0x22 : 0x21; // sampling ratio (h,v)
  *ptr++ = 0x00; // quant table id
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = numQtables == 1 ? 0x00 : 0x01; // quant table id
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // sampling ratio (h,v)
  *ptr++ = 0x01; // quant table id
  segment.setStatus(JpegSegment::Recalculated);
}

void ConstructSos(JpegSegment& segment)
{
  unsigned char *ptr = segment.reserveForAccess(MARKER_SOS, 12);
  *ptr++ = 0x03; // number of components
  *ptr++ = 0x01; // id of component
  *ptr++ = 0x00; // huffman table id (DC, AC)
  *ptr++ = 0x02; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x03; // id of component
  *ptr++ = 0x11; // huffman table id (DC, AC)
  *ptr++ = 0x00; // start of spectral
  *ptr++ = 0x3F; // end of spectral
  *ptr++ = 0x00; // successive approximation bit position (high, low)
  segment.setStatus(JpegSegment::Recalculated);
}

void CopyQuantization(JpegSegment& segment0, JpegSegment& segment1, const unsigned char* p)
{
  unsigned char b = 0;
  segment0.initForAdd(MARKER_DQT, 67);
  segment0.addData(&b, 1);
  segment0.addData(p, 64);
  b = 1;
  segment1.initForAdd(MARKER_DQT, 67);
  segment1.addData(&b, 1);
  segment1.addData(p + 64, 64);
  segment0.setStatus(JpegSegment::Received);
  segment1.setStatus(JpegSegment::Received);
}

