﻿///////////////////////////////////////////////////////////////////////////
//!  @author        Alexander Ryltsov
////
#define _CRTDBG_MAP_ALLOC
#include <NetCommon.h>
#include <stdlib.h>
#include <crtdbg.h>
#include "LiveCapsule.h"
#include "playbacksupport.h"
#include <time.h>
#include <map>
#include <liveMedia.hh>

//extern bool CheckMemoryLeaks();

void CFrameQueue::Push(FrameInfo* f)
{
  CLocker l(Door);
  PassedCount++;
  if (Frames.size() < MaxSize) {
    Frames.push_back(f);
  }
  else {
    FrameInfo::Release(f);
  }
}

FrameInfo* CFrameQueue::Peek()
{
  FrameInfo* f = NULL;
  CLocker l(Door);
  if (!Frames.empty()) {
    f = Frames.front();
  };
  return f;
}

FrameInfo* CFrameQueue::Pop()
{
  FrameInfo* f = NULL;
  CLocker l(Door);
  if (!Frames.empty()) {
    f = Frames.front();
    Frames.pop_front();
    PlayedCount++;
  };
  return f;
}

int CFrameQueue::GetCount()
{
  CLocker l(Door);
  return Frames.size();
}

class CapsuleMediaSubsession : public IMediaSubsession
{
  SChannelSettings &Settings;
  bool Audio;
public:
  MediaSubsession * MediaSub;
  CapsuleMediaSubsession(SChannelSettings &Settings, bool Audio) : Settings(Settings), Audio(Audio), MediaSub(NULL) {}
  virtual unsigned short   videoWidth()  const { return Settings.Width; }
  virtual unsigned short   videoHeight() const { return Settings.Height; }
  virtual unsigned         videoFPS()    const { return Settings.FPS; }
  virtual unsigned         numChannels() const { return Settings.Channels; }
  virtual char const*      mediumName()  const { return Audio ? "audio" : "video"; }
  virtual char const*      codecName()   const { return Settings.Codec.c_str(); }
  virtual char const*      fmtp_config() const { return ""; }
  virtual char const*      fmtp_spropparametersets() const { return ""; }
  virtual unsigned         rtpTimestampFrequency()   const { return Audio ? 8000 : 90000; }
  virtual MediaSubsession* mediaSub() { return MediaSub; }
};


bool CLiveCapsule::ConfigureWrapper()
{
  StepLog.SetObjectId(FullConfig.TestSequence.ObjectId);

  Live555Wrapper.AttachNotify(NotifyFunctionSend, NotifyFunctionReceive, NotifyFunctionLog, this);

  if (FullConfig.TestSequence.UseVideo) {
    Live555Wrapper.AttachChannel(ELC_Video, FullConfig.Video.Codec.c_str(), VideoDataFunction, this,
      FullConfig.Video.MulticastAddress.c_str(), FullConfig.Video.MulticastRtpPort, FullConfig.Video.MulticastTTL);
    FramesVideo.Subsession = new CapsuleMediaSubsession(FullConfig.Video, false);
    FramesVideo.Format = ProbeByCodec(FramesVideo.Subsession);
    if (!FramesVideo.Format) {
      StepLog.AddLogText((std::string("Warning: Unknow video format ") + FramesVideo.Subsession->codecName()).c_str());
      return false;
    }
  };
  if (FullConfig.TestSequence.UseAudio) {
    Live555Wrapper.AttachChannel(ELC_Audio, FullConfig.Audio.Codec.c_str(), AudioDataFunction, this,
      FullConfig.Audio.MulticastAddress.c_str(), FullConfig.Audio.MulticastRtpPort, FullConfig.Audio.MulticastTTL);
    FramesAudio.Subsession = new CapsuleMediaSubsession(FullConfig.Audio, true);
    FramesAudio.Format = ProbeByCodec(FramesAudio.Subsession);
    if (!FramesAudio.Format) {
      StepLog.AddLogText((std::string("Warning: Unknow audio format ") + FramesAudio.Subsession->codecName()).c_str());
      return false;
    }
  };
  if (FullConfig.TestSequence.UseMetadata) {
    Live555Wrapper.AttachChannel(ELC_Metadata, FullConfig.Metadata.Codec.c_str(), MetadataDataFunction, this);
    if (!FramesVideo.Subsession) {
      FramesVideo.Subsession = new CapsuleMediaSubsession(FullConfig.Metadata, false);
      FramesVideo.Format = ProbeByCodec(FramesVideo.Subsession);
    }
  };

  if (FullConfig.TestSequence.UseBackchannel) {
    Live555Wrapper.AttachChannel(ELC_AudioBack, FullConfig.Backchannel.Codec.c_str(),
      NULL, this, FullConfig.Backchannel.MulticastAddress.c_str());
  };

  Live555Wrapper.AttachCompletion(CompleteFunction, this);

  Live555Wrapper.AttachCredentials(FullConfig.ControlConnection.User.c_str(), FullConfig.ControlConnection.Password.c_str());

  if (FullConfig.ControlConnection.ForceDigest) {
    Live555Wrapper.ForceDigest();
  }

  Live555Wrapper.SetBase64LineBreaks(FullConfig.TestSequence.Base64LineBreaks);

  Live555Wrapper.Open(FullConfig.ControlConnection.Uri.c_str(), FullConfig.ControlConnection.TransportUri.c_str(), FullConfig.ControlConnection.Transport, FullConfig.ControlConnection.Port);

  Live555Wrapper.SetBackchannelProperties(FullConfig.Backchannel.UsePostChannel);

  PlayType = FullConfig.TestSequence.SequenceNumber;
  if (PlayType < 0) PlayType = 0;
  if (PlayType > SequenceCount) PlayType = 0;

  return true;
}

void CLiveCapsule::NotifyFunctionSend(const void* UserData, ELiveStep Step, const char* MessageData)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  pL->StepLog.SetRequest(MessageData);
}

void CLiveCapsule::NotifyFunctionReceive(const void* UserData, ELiveStep Step, const char* MessageData)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  pL->LastResponse = MessageData;
  pL->StepLog.SetResponse(MessageData);
}

void CLiveCapsule::NotifyFunctionLog(const void* UserData, ELiveStep Step, const char* MessageData)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  pL->StepLog.AddLogText(MessageData);
}
#if 1
void CLiveCapsule::VideoDataFunction(
  const void* UserData,
  unsigned char* ReceiveBuffer,
  unsigned FrameSize,
  unsigned NumTruncatedBytes,
  struct timeval PresentationTime,
  unsigned DurationInMicroseconds
)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  FrameInfo *Info = pL->FramesVideo.Format->NextChunk(
    pL->FramesVideo.Subsession, ReceiveBuffer, FrameSize, pL->FramesVideo.Prefix);
  Info->frameHead.FrameType = pL->FramesVideo.Format->AsFOURCC();
  Info->frameHead.Start =
    ((REFERENCE_TIME)PresentationTime.tv_sec) * 10000000 +
    ((REFERENCE_TIME)PresentationTime.tv_usec) * 10;
  pL->FramesVideo.Push(Info);
}
#else
void CLiveCapsule::VideoDataFunction(
  const void* UserData,
  unsigned char* ReceiveBuffer,
  unsigned FrameSize,
  unsigned NumTruncatedBytes,
  struct timeval PresentationTime,
  unsigned DurationInMicroseconds
)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  if (pL->FramesVideo.frameBuffer.size()
    && (pL->FramesVideo.t_sec != PresentationTime.tv_sec || pL->FramesVideo.t_usec != PresentationTime.tv_usec)) {
    FrameInfo *Info = pL->FramesVideo.Format->NextChunk(
      pL->FramesVideo.Subsession, &pL->FramesVideo.frameBuffer[0], pL->FramesVideo.frameBuffer.size(), pL->FramesVideo.Prefix);
    Info->frameHead.FrameType = pL->FramesVideo.Format->AsFOURCC();
    Info->frameHead.Start =
      ((REFERENCE_TIME)pL->FramesVideo.t_sec) * 10000000 +
      ((REFERENCE_TIME)pL->FramesVideo.t_usec) * 10;

    pL->FramesVideo.Push(Info);

    pL->FramesVideo.frameBuffer.resize(0);
  }

  FrameInfo *Info = pL->FramesVideo.Format->NextChunk(
    pL->FramesVideo.Subsession, ReceiveBuffer, FrameSize, pL->FramesVideo.Prefix);

  pL->FramesVideo.frameBuffer.insert(pL->FramesVideo.frameBuffer.end(), (uint8_t*)Info->pdata, (uint8_t*)Info->pdata + Info->frameHead.FrameLen);

  pL->FramesVideo.t_sec = PresentationTime.tv_sec;
  pL->FramesVideo.t_usec = PresentationTime.tv_usec;

  FrameInfo::Release(Info);
}
#endif
void CLiveCapsule::AudioDataFunction(
  const void* UserData,
  unsigned char* ReceiveBuffer,
  unsigned FrameSize,
  unsigned NumTruncatedBytes,
  struct timeval PresentationTime,
  unsigned DurationInMicroseconds
)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  FrameInfo *Info = pL->FramesAudio.Format->NextChunk(
    pL->FramesAudio.Subsession, ReceiveBuffer, FrameSize, pL->FramesAudio.Prefix);
  Info->frameHead.FrameType = pL->FramesAudio.Format->AsFOURCC();
  Info->frameHead.Start =
    ((REFERENCE_TIME)PresentationTime.tv_sec) * 10000000 +
    ((REFERENCE_TIME)PresentationTime.tv_usec) * 10;
  pL->FramesAudio.Push(Info);
}

void CLiveCapsule::MetadataDataFunction(
  const void* UserData,
  unsigned char* ReceiveBuffer,
  unsigned FrameSize,
  unsigned NumTruncatedBytes,
  struct timeval PresentationTime,
  unsigned DurationInMicroseconds
)
{
  CLiveCapsule* pL = ((CLiveCapsule*)UserData);
  if (!pL->HasVideo()) {
    FrameInfo *Info = pL->FramesVideo.Format->NextChunk(
      pL->FramesVideo.Subsession, ReceiveBuffer, FrameSize, pL->FramesVideo.Prefix);
    Info->frameHead.FrameType = pL->FramesVideo.Format->AsFOURCC();
#if 1
    Info->frameHead.Start =
      ((REFERENCE_TIME)PresentationTime.tv_sec) * 10000000 +
      ((REFERENCE_TIME)PresentationTime.tv_usec) * 10;
#else
    Info->frameHead.Start =
      ((REFERENCE_TIME)pL->FramesVideo.t_sec) * 10000000 +
      ((REFERENCE_TIME)pL->FramesVideo.t_usec) * 10;
#endif
    pL->FramesVideo.Push(Info);
  }

  FrameInfo* Info = FrameInfo::Init(FrameSize + 1);
  memcpy(Info->pdata, ReceiveBuffer, FrameSize);
  Info->pdata[FrameSize] = 0;
  pL->FramesMeta.Push(Info);
}

bool GetErrorString(std::string &str, int Error)
{
  LPVOID lpMsgBuf;
  FormatMessageW(
    FORMAT_MESSAGE_ALLOCATE_BUFFER |
    FORMAT_MESSAGE_FROM_SYSTEM |
    FORMAT_MESSAGE_IGNORE_INSERTS,
    NULL,
    Error,
    MAKELANGID(LANG_ENGLISH, SUBLANG_DEFAULT), // Default language
    (LPWSTR)&lpMsgBuf,
    0,
    NULL
  );
  if (lpMsgBuf) {
    char buffer[1024];
    WideCharToMultiByte(CP_UTF8, 0, (LPCWSTR)lpMsgBuf, -1, buffer, sizeof(buffer) - 1, NULL, NULL);
    str = buffer;
    LocalFree(lpMsgBuf);
    return true;
  }
  else {
    str.clear();
    return false;
  }
}

bool CLiveCapsule::GetSpecificErrorDescription(char *Buffer, ELiveStep Step, int &Result)
{
  if (Result == -10057) {
    if (((Step == ELS_Describe) || (Step == ELS_Options)) && (FullConfig.ControlConnection.Transport == ELT_HTTP)) {
      sprintf(Buffer, "Network error 10057 - probably HTTP tunnel is not working");
      return true;
    }
    if (Step == ELS_Teardown) {
      sprintf(Buffer, "Warning: Network error 10057 - probably device close connection before sending reply");
      StepLog.AddLogText(Buffer);
      Result = 0;
      return true;
    }
  }
  if (Result == RPC_E_TIMEOUT) {
    sprintf(Buffer, "Network error - timeout");
    return true;
  }
  if (Result == 401) {
    if (LastResponse.find("WWW-Authenticate: Basic") != std::string::npos) {
      sprintf(Buffer, "RTSP error - Basic Authentication is not allowed");
    }
    else if (LastResponse.find("WWW-Authenticate:") != std::string::npos) {
      sprintf(Buffer, "RTSP error 401 - probably device is not accepting our authentication data");
    }
    else {
      sprintf(Buffer, "Error: Digest authentication is mandatory for Profile T.");
    }
    return true;
  }
  return false;
}

void CLiveCapsule::CompleteFunction(const void* UserData, ELiveStep Step, int Result, const char* Body)
{
  char Buffer[1024];
  Buffer[0] = 0;
  if (Result) {
    if (!((CLiveCapsule*)UserData)->GetSpecificErrorDescription(Buffer, Step, Result)) {
      if (Result < 0) {
        std::string ErrorText;
        if (GetErrorString(ErrorText, -Result)) {
          sprintf_s(Buffer, sizeof(Buffer) - 1, "Network error %i: %s", -Result, ErrorText.c_str());
        }
        else {
          sprintf_s(Buffer, sizeof(Buffer) - 1, "Network error %i", -Result);
        }
      }
      else if (Result == 555) {
        sprintf_s(Buffer, sizeof(Buffer) - 1, "Unmanaged exception in %s", Body);
      }
      else {
        sprintf_s(Buffer, sizeof(Buffer) - 1, "Protocol error %i in %s", Result, Body);
      }
    }
  }
  ((CLiveCapsule*)UserData)->StepLog.SetStepResult(Result, Buffer);
}

bool CLiveCapsule::Play()
{
  if (PlayType) {
    return RunSequence(PlayType);
  }
  else {
    return PlayAsFilter();
  }
}

void CLiveCapsule::Pause()
{
  if (!PlayType) {
    if (GetCurrentStep() == ELS_Play) {
      StepLog.CreateNewStep("Pause");
      Live555Wrapper.Pause();
      Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout);
    }
  }
}

void CLiveCapsule::Stop()
{
  if ((GetCurrentStep() >= ELS_Play) && (GetCurrentStep() < ELS_Teardown)) {
    StepLog.CreateNewStep("Teardown");
    Live555Wrapper.Teardown();
    Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout);
  }
}

bool CLiveCapsule::PlayAsFilter()
{
  if (!Prepare(FullConfig.ControlConnection.Timeout)) {
    return false;
  }
  StepLog.CreateNewStep("Play");
  Live555Wrapper.Play();
  Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout);
  return true;
}

bool CLiveCapsule::FullStop()
{
  StepLog.CreateNewStep("Teardown");
  Live555Wrapper.Teardown();
  return !Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout);
}

bool CLiveCapsule::PlayAsTestBase(int SleepAfter)
{
  if (!Prepare(FullConfig.ControlConnection.Timeout)) {
    return false;
  }
  StepLog.CreateNewStep("Play");
  Live555Wrapper.Play();
  if (Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout))
    return false;
  Live555Wrapper.PlaySinks();
  Sleep(SleepAfter);
  return true;
}

bool CLiveCapsule::Configure(const WCHAR* FileName)
{
  FullConfig.Load(FileName);
  return ConfigureWrapper();
}

bool CLiveCapsule::HasAudio() const
{
  return FullConfig.TestSequence.UseAudio;
}

bool CLiveCapsule::HasVideo() const
{
  return FullConfig.TestSequence.UseVideo;
}

bool CLiveCapsule::HasMeta() const
{
  return FullConfig.TestSequence.UseMetadata;
}

bool CLiveCapsule::IsFakeVideo() const
{
  return FullConfig.TestSequence.UseMetadata && !FullConfig.TestSequence.UseVideo;
}

RTSPClientState CLiveCapsule::GetState() const
{
  return RTSP_STATE_IDLE;
}

bool CLiveCapsule::GetVideoFormat(CMediaType *pmt)
{
  if (!FramesVideo.Format || !FramesVideo.Subsession)
    return false;
  return FramesVideo.Format->GetFormat(pmt, FramesVideo.Subsession, &FramesVideo.Prefix);
}

bool CLiveCapsule::CheckVideoFormat(const CMediaType *pmt) const
{
  if (!FramesVideo.Format || !FramesVideo.Subsession)
    return false;
  return FramesVideo.Format->CheckFormat(pmt, FramesVideo.Subsession);
}

bool CLiveCapsule::GetAudioFormat(CMediaType *pmt)
{
  if (!FramesAudio.Format || !FramesAudio.Subsession)
    return false;
  return FramesAudio.Format->GetFormat(pmt, FramesAudio.Subsession, &FramesAudio.Prefix);
}

bool CLiveCapsule::CheckAudioFormat(const CMediaType *pmt) const
{
  if (!FramesAudio.Format || !FramesAudio.Subsession)
    return false;
  return FramesAudio.Format->CheckFormat(pmt, FramesAudio.Subsession);
}

FrameInfo* CLiveCapsule::GetTop(bool Audio)
{
  if (Audio) {
    return FramesAudio.Pop();
  }
  else {
    return FramesVideo.Pop();
  };
}

int CLiveCapsule::GetCount(bool Audio)
{
  if (Audio) {
    return FramesAudio.GetCount();
  }
  else {
    return FramesVideo.GetCount();
  };
}

FrameInfo* CLiveCapsule::GetMeta()
{
  return FramesMeta.Pop();
}

void CLiveCapsule::InterruptableSleep(int Timeout)
{
  InSleep = true;
  int Count = Timeout / 100;
  while (InSleep && !Terminating && (Count-- > 0)) {
    Sleep(100);
  };
  InSleep = false;
}

void CLiveCapsule::ExtractResolutionFromSdp()
{
  if (strcmp(FramesVideo.Subsession->mediaSub()->codecName(), "MP4V-ES") == 0) {
    if (const char *fmtp_config = FramesVideo.Subsession->mediaSub()->fmtp_config()) {
      unsigned char buffer[128];
      unsigned size = 0;
      unsigned i = 0;
      if (const char *vol_header = strstr(fmtp_config, "00000120")) {
        size = strlen(vol_header) / 2;
        for (i = 0; i < size; ++i) {
          int val = 0;
          if (1 != sscanf(&vol_header[2 * i], "%2X", &val)) break;
          buffer[i] = (unsigned char)val;
        }
      }
      if (i == size && size) {
        ParseMPEG4VOL(buffer, size, FramesVideo.Subsession->mediaSub()->readSource());
      }
    }
  }
  else if (strcmp(FramesVideo.Subsession->mediaSub()->codecName(), "H264") == 0) {
    unsigned int num = 0;
    SPropRecord *SPropRecords = parseSPropParameterSets(FramesVideo.Subsession->mediaSub()->fmtp_spropparametersets(), num);
    if (SPropRecords && num > 0) {
      ParseH264SPS(SPropRecords[0].sPropBytes, SPropRecords[0].sPropLength, FramesVideo.Subsession->mediaSub()->readSource());
    }
  }
  else if (strcmp(FramesVideo.Subsession->mediaSub()->codecName(), "H265") == 0) {
    MediaSubsession* sub = FramesVideo.Subsession->mediaSub();
    SPropRecord* sPropRecords[3];
    unsigned numSPropRecords[3];
    sPropRecords[0] = parseSPropParameterSets(sub->fmtp_spropvps(), numSPropRecords[0]);
    sPropRecords[1] = parseSPropParameterSets(sub->fmtp_spropsps(), numSPropRecords[1]);
    sPropRecords[2] = parseSPropParameterSets(sub->fmtp_sproppps(), numSPropRecords[2]);

    if (sPropRecords[1] && numSPropRecords[1] > 0) {
      ParseH265SPS(sPropRecords[1][0].sPropBytes, sPropRecords[1][0].sPropLength, FramesVideo.Subsession->mediaSub()->readSource());
    }
  }
}

bool CLiveCapsule::CheckResolution()
{
  char Buffer[100];
  StepLog.CreateNewStep("Checking actual resolution");
  sprintf(Buffer, "Requested resolution: %u x %u", FullConfig.Video.Width, FullConfig.Video.Height);
  StepLog.AddLogText(Buffer);
  int Width = 0;
  int Height = 0;

  if (FramesVideo.Subsession->mediaSub()) {
    if (FramesVideo.Subsession->mediaSub()->readSource()->IsFrameDimensionsParsed()) {
      Width = FramesVideo.Subsession->mediaSub()->readSource()->fFrameWidth;
      Height = FramesVideo.Subsession->mediaSub()->readSource()->fFrameHeight;
    }

    if (Width <= 0 || Height <= 0)
    {
      ExtractResolutionFromSdp();
      Width = FramesVideo.Subsession->mediaSub()->readSource()->fFrameWidth;
      Height = FramesVideo.Subsession->mediaSub()->readSource()->fFrameHeight;
    }
  }
  sprintf(Buffer, "Detected resolution: %u x %u", Width, Height);
  StepLog.AddLogText(Buffer);
  int Tolerance = 1;
  if (FullConfig.Video.Codec == "JPEG") {
    Tolerance = 8;
  }
  if (FullConfig.Video.Width != Width || FullConfig.Video.Height != Height) {
    StepLog.AddLogText("Warning: Resolution not matching exactly!");
  }
  if (!(Width >= FullConfig.Video.Width - Tolerance && Width <= FullConfig.Video.Width + Tolerance) ||
    !(Height >= FullConfig.Video.Height - Tolerance && Height <= FullConfig.Video.Height + Tolerance)
    ) {
    StepLog.SetStepResult(1, "Verification failed!");
    return false;
  }
  StepLog.SetStepResult();
  return true;
}

bool GetChannelPackets(CLive555Wrapper& wrap, ELiveChannels Channel, int& Packets)
{
  Packets = 0;

  RTPSource* source = wrap.GetChannelSource(Channel);
  if (!source)
    return false;

  RTPReceptionStatsDB& db = source->receptionStatsDB();
  Packets = db.totNumPacketsReceived();
  return true;
}

bool CLiveCapsule::CheckFrames(bool MakeStep, int Time)
{
  char Buffer[100];
  bool Ret = true;
  if (MakeStep) {
    StepLog.CreateNewStep("Checking media frames count");
  }
  if (Time < 1000) Time = 1000;
  if (FullConfig.TestSequence.UseVideo) {
    sprintf(Buffer, "%i video frames captured (%.1f FPS), %i played", FramesVideo.GetPassed(), double(FramesVideo.GetPassed()) * 1000 / Time, FramesVideo.GetPlayed());
    StepLog.AddLogText(Buffer);
    if (FramesVideo.GetPassed() <= 0) {
      Ret = false;
      int Packets;
      if (GetChannelPackets(Live555Wrapper, ELC_Video, Packets)) {
        if (Packets > 0) {
          sprintf(Buffer, "%i RTP packet(s) captured, maybe they are wrong formatted or network is noisy", Packets);
          StepLog.AddLogText(Buffer);
        }
      }
    };
    if ((FramesVideo.GetPlayed() <= 0) && (FramesVideo.GetPassed() > 0)) {
      StepLog.AddLogText("Warning: maybe video decoding filters are not active");
    };
  }
  if (FullConfig.TestSequence.UseAudio) {
    sprintf(Buffer, "%i audio frames captured, %i played", FramesAudio.GetPassed(), FramesAudio.GetPlayed());
    StepLog.AddLogText(Buffer);
    if (FramesAudio.GetPassed() <= 0) {
      Ret = false;
      int Packets;
      if (GetChannelPackets(Live555Wrapper, ELC_Audio, Packets)) {
        if (Packets > 0) {
          sprintf(Buffer, "%i RTP packet(s) captured, maybe they are wrong formatted or network is noisy", Packets);
          StepLog.AddLogText(Buffer);
        }
      }
    };
    if ((FramesAudio.GetPlayed() <= 0) && (FramesAudio.GetPassed() > 0)) {
      StepLog.AddLogText("Warning: maybe audio decoding filters are not active");
    };
  }
  if (FullConfig.TestSequence.UseMetadata) {
    sprintf(Buffer, "%i metadata frames captured", FramesMeta.GetPassed());
    StepLog.AddLogText(Buffer);
    if (FramesMeta.GetPassed() <= 0) {
      Ret = false;
      int Packets;
      if (GetChannelPackets(Live555Wrapper, ELC_Metadata, Packets)) {
        if (Packets > 0) {
          sprintf(Buffer, "%i RTP packet(s) captured, maybe they are wrong formatted or network is noisy", Packets);
          StepLog.AddLogText(Buffer);
        }
      }
    };
  }
  if (MakeStep) {
    if (Ret) {
      StepLog.SetStepResult();
    }
    else {
      StepLog.SetStepResult(1, "No frames captured");
    }
  }
  return Ret;
}

bool CLiveCapsule::CheckOptions(unsigned int Timeout)
{
  const char* Body;
  if (FullConfig.TestSequence.CheckOptions) {
    StepLog.CreateNewStep("Options");
    Live555Wrapper.Options();
    if (Live555Wrapper.WaitResult(Body, Timeout))
      return false;

    char Buffer[512];
    sprintf_s(Buffer, 511, "Options [%s]", Body);
    StepLog.CreateNewStep("Checking Options");
    StepLog.AddLogText(Buffer);
    bool Ret = true;
#define CheckOptionsParam(param)											\
    if (!strstr(Body, param)) {												\
		if (param == "SET_PARAMETER")										\
		{																	\
			sprintf(Buffer, "RECOMMENDED RTSP method missed [%s]", param);	\
			StepLog.AddLogText(Buffer);										\
		}																	\
		else																\
		{																	\
			sprintf(Buffer, "Missed [%s]", param);							\
			StepLog.AddLogText(Buffer);										\
			Ret = false;													\
		}																	\
       }
    CheckOptionsParam("SET_PARAMETER");
    CheckOptionsParam("DESCRIBE");
    CheckOptionsParam("SETUP");
    CheckOptionsParam("PLAY");
    CheckOptionsParam("TEARDOWN");

    if (Ret) {
      StepLog.SetStepResult();
    }
    else {
      StepLog.SetStepResult(1, "Some options missed");
    }
    return Ret;
  }
  else {
    return true;
  }
}

bool CLiveCapsule::Prepare(unsigned int Timeout)
{
  if ((GetCurrentStep() < ELS_Play) || (GetCurrentStep() >= ELS_Teardown)) {

    if (!CheckOptions(Timeout))
      return false;

    const char* Body;

    StepLog.CreateNewStep("Describe");
    Live555Wrapper.Describe();
    if (Live555Wrapper.WaitResult(Body, Timeout))
      return false;

    StepLog.CreateNewStep("Create Media Session");
    if (!Live555Wrapper.CreateMediaSessionBySdp(Body, false)) {
      StepLog.SetStepResult(1, "Error creating media session");
      return false;
    }
    StepLog.SetStepResult();

    int Channels = 0;
    if (FullConfig.TestSequence.UseVideo) {
      ((CapsuleMediaSubsession*)FramesVideo.Subsession)->MediaSub = Live555Wrapper.GetChannelSession(ELC_Video);
      Channels++;
    }
    if (FullConfig.TestSequence.UseAudio) Channels++;
    if (FullConfig.TestSequence.UseMetadata) Channels++;
    if (FullConfig.TestSequence.UseBackchannel) Channels++;
    for (int i = 0; i < Channels; i++) {
      StepLog.CreateNewStep("Setup");
      Live555Wrapper.SetupSub(i);
      if (Live555Wrapper.WaitResult(Timeout))
        return false;
    }

    StepLog.CreateNewStep("Create Sinks");
    if (!Live555Wrapper.CreateSinks()) {
      StepLog.SetStepResult(1, "Error creating sinks");
      return false;
    }
    StepLog.SetStepResult();
  }
  return true;
}

bool CLiveCapsule::PrepareRtp()
{
  if ((GetCurrentStep() < ELS_Play) || (GetCurrentStep() >= ELS_Teardown)) {
    StepLog.CreateNewStep("Create Media Session");
    if (!Live555Wrapper.CreateRtpSubsessionsByChannels()) {
      StepLog.SetStepResult(1, "Error creating media session");
      return false;
    }
    StepLog.SetStepResult();

    StepLog.CreateNewStep("Create Sinks");
    if (!Live555Wrapper.CreateSinks()) {
      StepLog.SetStepResult(1, "Error creating sinks");
      return false;
    }
    StepLog.SetStepResult();
  }
  return true;
}

bool CLiveCapsule::CheckTermination()
{
  if (Terminating) {
    if (Halt) {
      StepLog.SetStepResult(0, "Forced stop - halt button pressed");
    }
    else {
      StepLog.SetStepResult(0, "Forced stop - rolling back");
    }
    return false;
  }
  else {
    StepLog.SetStepResult();
    return true;
  }
}

bool CLiveCapsule::PlayAsTest1TimeWait()
{
  if (!PlayAsTestBase(0))
    return false;

  int Start = GetTickCount();
  StepLog.CreateNewStep("Waiting for 10 seconds");
  InterruptableSleep(10000);
  if (!CheckTermination())
    return false;

  if (!CheckFrames(true, GetTickCount() - Start)) {
    return false;
  }

  FullStop();
  return true;
}

static
void PRtpJPEGExtHdrCallback(uint16_t profile, uint16_t seq, uint16_t len, uint8_t* pHdrData, void* pPriv)
{
  if (pPriv && (profile == 0xFFD8 || profile == 0xFFFF))
  {
    char Buffer[256];
    CJPEGExtensionFrames* video = (CJPEGExtensionFrames*)pPriv;
    u_int32_t SpecialHeaderStatus = *((u_int32_t*)pHdrData);

    if (video->Subsession != nullptr && video->Subsession->mediaSub()) {
      MediaSubsession* sub = video->Subsession->mediaSub();
      unsigned int FrameWidth = sub->rtpSource()->fFrameWidth;
      unsigned int FrameHeight = sub->rtpSource()->fFrameHeight;

      if (SpecialHeaderStatus == 2 && *((u_int32_t*)pHdrData) == 1)
      {
        ++video->FailedPacketsReceived;
        sprintf(Buffer, "SeqNum=%u Incorrect header!", htons(seq));
        video->Log.AddLogText(Buffer);
      }
      if (SpecialHeaderStatus == 1)
      {
        ++video->PacketsReceived;
        //sprintf(Buffer, "SeqNum=%u Resolution: %u x %u", htons(seq), FrameWidth, FrameHeight);
        //video->Log.AddLogText(Buffer);
      }
      //sub->rtpSource()->fFrameWidth = 0;
      //sub->rtpSource()->fFrameHeight = 0;
    }
  }
}

bool CLiveCapsule::PlayAsTest2FramesWait()
{

  if (!Prepare(FullConfig.ControlConnection.Timeout)) {
    return false;
  }

  CJPEGExtensionFrames JPEGFrames(FramesVideo.Subsession, StepLog);
  if (FullConfig.TestSequence.CheckJPEGExtension) {
    Live555Wrapper.AttachRtpCallback(ELC_Video, PRtpJPEGExtHdrCallback, &JPEGFrames);
  }

  StepLog.CreateNewStep("Play");
  Live555Wrapper.Play();
  if (Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout))
    return false;
  Live555Wrapper.PlaySinks();
  Sleep(0);

  int MaxFrames = 100;
  int Frames = MaxFrames;
  int WaitingTimeout = FullConfig.TestSequence.Timeout;
  int CriticalFrames = WaitingTimeout * FullConfig.Video.FPS / 2000;
  if (Frames > CriticalFrames) {
    if (CriticalFrames < 3) {
      CriticalFrames = 3;
      if (FullConfig.Video.FPS >= 1) {
        WaitingTimeout = CriticalFrames * 2000 / FullConfig.Video.FPS;
      }
      else {
        WaitingTimeout = CriticalFrames * 2000;
      }
    }
    Frames = CriticalFrames;
  }
  char Buffer[128];
  sprintf(Buffer, "Waiting for %i frames up to %i ms ", Frames, WaitingTimeout);
  StepLog.CreateNewStep(Buffer);
  int Start = GetTickCount();
  int End = Start + WaitingTimeout;
  while (!Terminating && (FramesVideo.GetPassed() < Frames)) {
    if ((int)GetTickCount() > End) {
      int Time = GetTickCount() - Start;
      ///int Time = FullConfig.TestSequence.Timeout;
      if (Time < 1000) Time = 1000;
      double FPS = double(FramesVideo.GetPassed()) * 1000 / Time;
      sprintf(Buffer, "Only %i frames captured (%.1f FPS)", FramesVideo.GetPassed(), FPS);
      StepLog.AddLogText(Buffer);
      if (FPS > 0.001) {
        int TimeoutHint = (int)(MaxFrames / FPS + 2);
        sprintf(Buffer, "Please increase Operation Delay to %i ms in case device is working properly!", TimeoutHint * 1000);
        StepLog.AddLogText(Buffer);
      }
      else {
        StepLog.AddLogText("Please increase Operation Delay in case device is working properly!");
      }
      int Packets;
      if (GetChannelPackets(Live555Wrapper, ELC_Video, Packets)) {
        if (Packets > 0) {
          sprintf(Buffer, "%i RTP packet(s) captured, maybe they are wrong formatted or network is noisy", Packets);
          StepLog.AddLogText(Buffer);
        }
      }
      StepLog.SetStepResult(1, "Frames waiting timeout.");
      return false;
    }
    Sleep(100);
  };
  CheckFrames(false, GetTickCount() - Start);
  if (!CheckTermination())
    return false;

  if (FullConfig.TestSequence.CheckActualResolution) {
    if (!CheckResolution()) {
      return false;
    }
  }

  Live555Wrapper.AttachRtpCallback(ELC_Video, nullptr, nullptr);
  if (FullConfig.TestSequence.CheckJPEGExtension) {
    StepLog.CreateNewStep("Checking extension packets");

    if (FullConfig.Video.Height > 2040 || FullConfig.Video.Width > 2040)
    {
      if (JPEGFrames.PacketsReceived <= 0)
      {
        sprintf(Buffer, "No JPEG header extensions found in packets for resolution %u x %u", FullConfig.Video.Width, FullConfig.Video.Height);
        StepLog.AddLogText(Buffer);
        StepLog.SetStepResult(1, "Verification failed!");
        return false;
      }
      else if (JPEGFrames.FailedPacketsReceived > 10)
      {
        sprintf(Buffer, "Packets without JPEG header extensions found for resolution %u x %u", FullConfig.Video.Width, FullConfig.Video.Height);
        StepLog.AddLogText(Buffer);
        StepLog.SetStepResult(1, "Verification failed!");
        return false;
      }
      sprintf(Buffer, "%u Packets with JPEG header extensions found", JPEGFrames.PacketsReceived);
      StepLog.AddLogText(Buffer);
    }
    else
    {
      StepLog.AddLogText("Low resolution");
    }
    StepLog.SetStepResult();
    if (!CheckResolution()) {
      return false;
    }
  }

  FullStop();
  return true;
}

bool CLiveCapsule::PlayAsTest3TimeWaitRtp()
{
  if (!PrepareRtp())
    return false;

  int Start = GetTickCount();
  StepLog.CreateNewStep("Waiting for 10 seconds");
  InterruptableSleep(10000);
  if (!CheckTermination())
    return false;

  if (!CheckFrames(true, GetTickCount() - Start)) {
    return false;
  }

  //FullStop();
  return true;
}

bool CLiveCapsule::PlayAsTest4Backchannel()
{
  Live555Wrapper.SetAdditionalFields("Require: www.onvif.org/ver20/backchannel\r\n");
  if (!PlayAsTestBase(0))
    return false;

  int Start = GetTickCount();
  StepLog.CreateNewStep("Waiting for 8 seconds");
  InterruptableSleep(8000);
  if (!CheckTermination())
    return false;

  FullStop();
  return true;
}

bool CLiveCapsule::PlayAsTest5Playback()
{
  int Timeout = FullConfig.ControlConnection.Timeout;

  ELiveChannels MasterChannel = ELC_Error;
  if (FullConfig.TestSequence.UseVideo) {
    MasterChannel = ELC_Video;
  }
  if (FullConfig.TestSequence.UseAudio && (MasterChannel == ELC_Error)) {
    MasterChannel = ELC_Audio;
  }
  if (FullConfig.TestSequence.UseMetadata && (MasterChannel == ELC_Error)) {
    MasterChannel = ELC_Metadata;
  }

  if ((GetCurrentStep() < ELS_Play) || (GetCurrentStep() >= ELS_Teardown)) {

    if (!CheckOptions(Timeout))
      return false;

    const char* Body;

    StepLog.CreateNewStep("Describe");
    Live555Wrapper.Describe();
    if (Live555Wrapper.WaitResult(Body, Timeout))
      return false;

    StepLog.CreateNewStep("Create Media Session");
    if (!Live555Wrapper.CreateMediaSessionBySdp(Body, true)) {
      StepLog.SetStepResult(1, "Error creating media session");
      return false;
    }
    StepLog.SetStepResult();


    Live555Wrapper.SetAdditionalFields(FullConfig.TestSequence.CustomSetupFields.c_str());

    int Channels = 0;
    if (FullConfig.TestSequence.UseVideo) {
      ((CapsuleMediaSubsession*)FramesVideo.Subsession)->MediaSub = Live555Wrapper.GetChannelSession(ELC_Video);
      Channels++;
    }
    if (FullConfig.TestSequence.UseAudio) Channels++;
    if (FullConfig.TestSequence.UseMetadata) Channels++;
    if (FullConfig.TestSequence.UseBackchannel) Channels++;
    for (int i = 0; i < Channels; i++) {
      StepLog.CreateNewStep("Setup");
      Live555Wrapper.SetupSub(i);
      if (Live555Wrapper.WaitResult(Timeout))
        return false;
    }

    StepLog.CreateNewStep("Create Sinks");
    if (!Live555Wrapper.CreateSinks()) {
      StepLog.SetStepResult(1, "Error creating sinks");
      return false;
    }
    StepLog.SetStepResult();
  }

  std::string &CustomPlayFields = FullConfig.TestSequence.CustomPlayFields;
  char Delim = CustomPlayFields.empty() ? 0 : CustomPlayFields[0];
  int pos = CustomPlayFields.find(Delim, 1);
  std::string CustomPlayFieldsLoc;
  if (pos > 0) {
    CustomPlayFieldsLoc = CustomPlayFields.substr(1, pos - 1);
    CustomPlayFields = CustomPlayFields.substr(pos);
  }
  else {
    if (!CustomPlayFields.empty()) {
      CustomPlayFieldsLoc = CustomPlayFields.substr(1);
      CustomPlayFields.clear();
    }
  }

  Live555Wrapper.SetAdditionalFields(CustomPlayFieldsLoc.c_str());

  PlaybackFrameLog fl(StepLog);
  Live555Wrapper.AttachRtpCallback(MasterChannel, PlaybackFrameLog::RtpExtHdrCallback, &fl);

  StepLog.CreateNewStep("Play");
  Live555Wrapper.Play(-1.0);
  Live555Wrapper.SetAdditionalFields(NULL);
  if (Live555Wrapper.WaitResult(Timeout))
    return false;

  int Start = GetTickCount();
  StepLog.CreateNewStep("Wait Stream");
  //  StepLog.CreateNewStep("Waiting for 10 seconds");
  InterruptableSleep(10000);
  Live555Wrapper.AttachRtpCallback(MasterChannel, NULL, NULL);

  char Buffer[128];
  sprintf(Buffer, "Frames received: %d", fl.GetRecords());
  StepLog.AddLogText(Buffer);
  if (!fl.GetRecords()) {
    StepLog.SetStepResult(1, "No frames captured");
    return false;
  }
  if (!CheckTermination())
    return false;

  return true;
}

bool CLiveCapsule::PlayAsTest6PlaybackTeardown()
{
  int Timeout = FullConfig.ControlConnection.Timeout;

  ELiveChannels MasterChannel = ELC_Error;
  if (FullConfig.TestSequence.UseVideo) {
    MasterChannel = ELC_Video;
  }
  if (FullConfig.TestSequence.UseAudio && (MasterChannel == ELC_Error)) {
    MasterChannel = ELC_Audio;
  }
  if (FullConfig.TestSequence.UseMetadata && (MasterChannel == ELC_Error)) {
    MasterChannel = ELC_Metadata;
  }

  if ((GetCurrentStep() < ELS_Play) || (GetCurrentStep() >= ELS_Teardown)) {

    if (!CheckOptions(Timeout))
      return false;

    const char* Body;

    StepLog.CreateNewStep("Describe");
    Live555Wrapper.Describe();
    if (Live555Wrapper.WaitResult(Body, Timeout))
      return false;

    StepLog.CreateNewStep("Create Media Session");
    if (!Live555Wrapper.CreateMediaSessionBySdp(Body, true)) {
      StepLog.SetStepResult(1, "Error creating media session");
      return false;
    }
    StepLog.SetStepResult();


    Live555Wrapper.SetAdditionalFields(FullConfig.TestSequence.CustomSetupFields.c_str());

    int Channels = 0;
    if (FullConfig.TestSequence.UseVideo) {
      ((CapsuleMediaSubsession*)FramesVideo.Subsession)->MediaSub = Live555Wrapper.GetChannelSession(ELC_Video);
      Channels++;
    }
    if (FullConfig.TestSequence.UseAudio) Channels++;
    if (FullConfig.TestSequence.UseMetadata) Channels++;
    if (FullConfig.TestSequence.UseBackchannel) Channels++;
    for (int i = 0; i < Channels; i++) {
      StepLog.CreateNewStep("Setup");
      Live555Wrapper.SetupSub(i);
      if (Live555Wrapper.WaitResult(Timeout))
        return false;
    }

    StepLog.CreateNewStep("Create Sinks");
    if (!Live555Wrapper.CreateSinks()) {
      StepLog.SetStepResult(1, "Error creating sinks");
      return false;
    }
    StepLog.SetStepResult();
  }

  std::string &CustomPlayFields = FullConfig.TestSequence.CustomPlayFields;
  char Delim = CustomPlayFields.empty() ? 0 : CustomPlayFields[0];
  int pos = CustomPlayFields.find(Delim, 1);
  std::string CustomPlayFieldsLoc;
  if (pos > 0) {
    CustomPlayFieldsLoc = CustomPlayFields.substr(1, pos - 1);
    CustomPlayFields = CustomPlayFields.substr(pos);
  }
  else {
    if (!CustomPlayFields.empty()) {
      CustomPlayFieldsLoc = CustomPlayFields.substr(1);
      CustomPlayFields.clear();
    }
  }

  Live555Wrapper.SetAdditionalFields(CustomPlayFieldsLoc.c_str());

  PlaybackFrameLog fl(StepLog);
  Live555Wrapper.AttachRtpCallback(MasterChannel, PlaybackFrameLog::RtpExtHdrCallback, &fl);

  StepLog.CreateNewStep("Play");
  Live555Wrapper.Play(-1.0);
  Live555Wrapper.SetAdditionalFields(NULL);
  if (Live555Wrapper.WaitResult(Timeout))
    return false;

  int Start = GetTickCount();
  StepLog.CreateNewStep("Wait Stream");
  //  StepLog.CreateNewStep("Waiting for 10 seconds");
  InterruptableSleep(10000);
  Live555Wrapper.AttachRtpCallback(MasterChannel, NULL, NULL);

  char Buffer[128];
  sprintf(Buffer, "Frames received: %d", fl.GetRecords());
  StepLog.AddLogText(Buffer);
  if (!fl.GetRecords()) {
    StepLog.SetStepResult(1, "No frames captured");
    return false;
  }
  if (!CheckTermination())
    return false;

  this->Stop();

  return true;
}


bool CLiveCapsule::PlayAsTest7TimeWaitAction()
{
  if (!PlayAsTestBase(0))
    return false;

  int Start = GetTickCount();
  StepLog.RunCommand("Action");
  StepLog.CreateNewStep("Waiting for 10 seconds");
  InterruptableSleep(10000);
  if (!CheckTermination())
    return false;

  if (!FullStop()) {
    return false;
  }

  if (!CheckFrames(true, GetTickCount() - Start)) {
    return false;
  }

  return true;
}



bool CLiveCapsule::RunStep(int Step)
{
  StepLog.CreateNewStep(GetLiveStepName(ELiveStep(Step)));
  return Live555Wrapper.RunCommand(ELiveStep(Step));
};

bool CLiveCapsule::RunStepSync(int Step)
{
  if (!RunStep(Step))
    return false;
  if (Live555Wrapper.WaitResult(FullConfig.ControlConnection.Timeout))
    return false;
  return true;
}

bool CLiveCapsule::RunSequence(int Sequence)
{
  if (IsInSequence())
    return false;
  if (Sequence < 0)
    return false;
  if (Sequence > SequenceCount)
    return false;
  InSequence = true;
  this->Sequence = Sequence;
  return true;
}

void CLiveCapsule::RunSequenceInternal(int Sequence)
{
  InSequence = true;
  try {
    switch (Sequence) {
    case 0: PlayAsFilter(); break;
    case 1: PlayAsTest1TimeWait(); break;
    case 2: PlayAsTest2FramesWait(); break;
    case 3: PlayAsTest3TimeWaitRtp(); break;
    case 4: PlayAsTest4Backchannel(); break;
    case 5: PlayAsTest5Playback(); break;
    case 6: PlayAsTest6PlaybackTeardown(); break;
    case 7: PlayAsTest7TimeWaitAction(); break;
    default:  break;
    }
  }
  catch (std::exception& e) {
    std::string error("LiveCapsule: ");
    error += e.what();
    CompleteFunction(this, GetCurrentStep(), 555, error.c_str());
  }
  catch (...) {
    CompleteFunction(this, GetCurrentStep(), 555, "LiveCapsule: Unknown exception");
  }
  InSequence = false;
}

bool CLiveCapsule::IsInSequence()
{
  return InSequence;
}

void CLiveCapsule::Thread()
{
  while (!Terminating) {
    if (Sequence) {
      RunSequenceInternal(Sequence);
      Sequence = 0;
    };
    Sleep(100);
  }
}

void CLiveCapsule::ClearMutators()
{
  InSleep = false;
  Live555Wrapper.AttachRtpCallback(ELC_Video, NULL, NULL);
  Live555Wrapper.SetAdditionalFields(NULL);
}

void TestSPS()
{
  unsigned char SPS[30] = {
    66, 1, 1, 1, 96, 0, 0, 3,
    0, 176, 0, 0, 3,
    0, 0, 3,
    0, 153, 160, 1, 0, 32, 6, 1, 99, 106, 164, 147, 47, 144
  };
  ParseH265SPS(SPS, 30, NULL);

}

void Testm4()
{
  FILE* file = fopen("C:\\Users\\alexanderr\\Documents\\customers\\ason\\ef_logs\\mpeg4\\mpeg4\\22.bin", "r");
  unsigned char buffer[1500];
  fread(buffer, 1, 1444, file);
  fclose(file);
  char f[20000];
  ParseMPEG4VOL(buffer, 1444, (FramedSource*)&f);
}

void TestMeNow()
{
  //Testm4();
	//TestSPS();
	CLiveCapsule live;
	//live.SetNotificationCallback(GlobalStepCallback);
	live.Configure(L"c:\\test.omsd1906RC2");
	live.RunSequence(2);
	Sleep(10000);
}

TSharpStepCallback GlobalStepCallback;
extern "C"   __declspec(dllexport)
int
__cdecl
SetGlobalStepCallback(TSharpStepCallback StepCallback)
{
  GlobalStepCallback = StepCallback;
  //TestMeNow();
  return 0;
}


extern "C"   __declspec(dllexport)
int
__cdecl
RunTestSession(const WCHAR* Filepath)
{
  _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
  {
    CLiveCapsule test;
    test.SetNotificationCallback(GlobalStepCallback);
    if (!test.Configure(Filepath)) {
      return 1;
    };
    if (!test.Play()) {
      return 2;
    }

    do {
      Sleep(1000);
    } while (test.IsInSequence());
    Sleep(1000);
  }

  //assert(CheckMemoryLeaks());

  return 0;
}

extern "C"   __declspec(dllexport)
int
__cdecl
StartTestSession(const WCHAR* Filepath)
{
  _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
  CLiveCapsule* p = new CLiveCapsule();
  p->SetNotificationCallback(GlobalStepCallback);
  if (!p->Configure(Filepath)) {
    return 0;
  };
  if (!p->Play()) {
    return 0;
  }
  /*do {
    Sleep(1000);
  } while(p->IsInSequence());*/
  Sleep(1000);

  return int(p);
}

extern "C"   __declspec(dllexport)
int
__cdecl
IsTestSequence(int Session)
{
  if (!Session) {
    return 0;
  }
  CLiveCapsule* p = (CLiveCapsule*)Session;
  return p->IsInSequence();
}

extern "C"   __declspec(dllexport)
int
__cdecl
ClearTestSequence(int Session)
{
  if (!Session) {
    return 0;
  }
  CLiveCapsule* p = (CLiveCapsule*)Session;
  p->ClearMutators();
  return 1;
}

extern "C"   __declspec(dllexport)
int
__cdecl
CallTestSequence(int Session, int Sequence)
{
  if (!Session) {
    return 0;
  }
  CLiveCapsule* p = (CLiveCapsule*)Session;
  p->RunSequence(Sequence);
  /* do {
     Sleep(1000);
   } while(p->IsInSequence());*/
  Sleep(1000);
  return 1;
}

extern "C"   __declspec(dllexport)
int
__cdecl
CallTestCommand(int Session, int Command)
{
  if (!Session) {
    return 0;
  }
  CLiveCapsule* p = (CLiveCapsule*)Session;
  return !p->RunStepSync(Command);
}

extern "C"   __declspec(dllexport)
int
__cdecl
CloseTestSession(int Session)
{
  if (!Session) {
    return 0;
  }
  CLiveCapsule* p = (CLiveCapsule*)Session;
  if (p->IsNeedTeardown()) {
    p->RunStepSync(ELS_Teardown);
  }
  delete p;
  //assert(CheckMemoryLeaks());
  return 0;
}

